/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */

package org.jboss.soa.esb.listeners.message;

import java.sql.Timestamp;

/**
 * Extraction of core from MessageCounter bean.
 * 
 * @author <a href="mailto:tcunning@redhat.com">tcunning@redhat.com</a> 
 *
 */
public class MessageCounterStatistics {


	private static final MessageCounterStatistics SINGLETON = new MessageCounterStatistics() ;
	
	private long lastSuccessfulMessageDate;
	private long lastFailedMessageDate;
	private int failedMessageCount;
	private int successMessageCount;
	private long totalProcessTime;
	private long bytesProcessed;
	private long bytesFailed;
	
	/**
	 * Constructor.
	 */
	public MessageCounterStatistics() {
		resetCounts();
	}
	
	/**
	 * Get the total number of failed messages.
	 * @return total number of failed messages
	 */
	public synchronized int getFailedMessageCount() {
		return failedMessageCount;
	}

	/**
	 * Get the time the last message was processed at.
	 * @return time the last message was processed at
	 */
	public synchronized String getLastSuccessfulMessageDate() {
		return getDate(lastSuccessfulMessageDate);
	}
	
	/**
	 * Get the time the last message was processed at.
	 * @return time the last message was processed at
	 */
	public synchronized String getLastFailedMessageDate() {
		return getDate(lastFailedMessageDate);
	}
	
	/**
	 * Get the total number of processed messages, both successful and failed.
	 * @return total number of processed messages
	 */
	public synchronized int getTotalMessageCount() {
		return (failedMessageCount + successMessageCount);
	}

	/**
	 * Get the total number of successfully processed messages.
	 * @return total number of successfully processed messages
	 */
	public synchronized int getSuccessfulMessageCount() {
		return successMessageCount;
	}

	public synchronized long getProcessedBytes() {
		return bytesProcessed;
	}
	
	public synchronized long getFailedBytes() {
		return bytesFailed;
	}
	
	/**
	 * Update the message counter based on the MessageStatusBean that is returned
	 * from the observable.
	 * @param o the observable object
	 * @param arg the MessageStatusBean 
	 */
	public synchronized void update(MessageStatusBean msb) {
		if (msb.getMessageStatus().equals(MessageStatusBean.MESSAGE_SENT)) {
			successMessageCount++;
			if (msb.getMessageTime() > lastSuccessfulMessageDate) {
				lastSuccessfulMessageDate = msb.getMessageTime();
			}
			bytesProcessed += msb.getMessageBytes();
		} else if (msb.getMessageStatus().equals(MessageStatusBean.MESSAGE_FAILED)) {
			failedMessageCount++;
			if (msb.getMessageTime() > lastFailedMessageDate) {
				lastFailedMessageDate = msb.getMessageTime();
			}
			bytesFailed += msb.getMessageBytes();
		}
		
		totalProcessTime += msb.getProcessTime();
    }

	/**
	 * Get the average time to process a message.
	 * @return average time to process a message 
	 */
	public synchronized Double getAverageSuccessTime() {
		if ((successMessageCount) > 0) {
			double ms = (double) totalProcessTime / (1000000 * successMessageCount);
			return new Double(ms);
		} else {
			return null;
		}
	}
	
	/**
	 * Reset the counts - this resets the totals and the last message time. 
	 */
	public synchronized void resetCounts() {
		lastSuccessfulMessageDate = 0;
		lastFailedMessageDate = 0;
		successMessageCount = 0;
		failedMessageCount = 0;
		totalProcessTime = 0;
		bytesProcessed = 0;
		bytesFailed = 0;
	}
	
	/**
	 * Get the string representation of the time.
	 * @param time The time.
	 * @return The string representation.
	 */
	private static String getDate(final long time) {
		if (time <= 0) {
			return "---" ;
		} else {
			// not sure why timestamp is being used.
			final Timestamp ts = new Timestamp(time) ;
			return ts.toString() ;
		}
	}
	
	public static MessageCounterStatistics getMessageCounterStatistics() {
		return SINGLETON ;
	}
}
