/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.message;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.message.LegacyMessageComposerAdapter;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.ConfigurationException;

/**
 * Message composer.
 * <p/>
 * A "Composer" is basically a "Builder" ala the GoF patterns.
 * <p/>
 * Implementations should be threadsafe (stateless) and must contain
 * a public default constructor.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public interface MessageComposer<T> {

    /**
     * Set the composer's configuration.
     *
     * @param config Composer configuration.
     * @throws ConfigurationException Bad configuration.
     */
    public void setConfiguration(ConfigTree config) throws ConfigurationException;

    /**
     * Compose an ESB "aware" message from the supplied message payload.
     * <p/>
     * Implementations need to construct and populate an ESB Message from the
     * messagePayload instance.
     *
     * @param messagePayload Message payload to be packaged, or a channel specific
     *                       container class for the message payload (e.g. a JMS message).
     * @return ESB aware message instance.
     * @throws MessageDeliverException Failed to compose message payload for delivery.
     */
    public Message compose(T messagePayload) throws MessageDeliverException;

    /**
     * Decompose an ESB "aware" message, extracting and returning the message payload.
     *
     * @param message ESB aware message instance to be decomposed.
     * @param originalInputMessagePayload The original input message payload used to
     * compose this (or ther original) message.  The original message can sometimes contain
     * information relevant during the decomposition process.  Whether or not this parameter
     * can be null depends on the MessageComposer implementation.
     * @return ESB unaware message instance - a message payload.
     * @throws MessageDeliverException Failed to decompose message payload.
     */
    public Object decompose(Message message, T originalInputMessagePayload) throws MessageDeliverException;

    /**
     * Uitility factory class for reflective {@link MessageComposer} construction.
     */
    public static class Factory {

        /**
         * Factory method.
         *
         * @param className Class name.
         * @param config    The composer configuration.
         * @return Composer instance.
         * @throws ConfigurationException  Bad configuration.
         * @throws MessageDeliverException Unable to construct composer.
         */
        public static MessageComposer getInstance(String className, ConfigTree config) throws ConfigurationException, MessageDeliverException {
            return getInstance(className, config, null);
        }

        /**
         * Factory method.
         *
         * @param className         Class name.
         * @param config            The composer configuration.
         * @param legacyAdapterType The {@link LegacyMessageComposerAdapter legacy adapter} to use if the configured
         *                          composer is not a {@link MessageComposer} instance.
         * @return Composer instance.
         * @throws ConfigurationException  Bad configuration.  Unable to create composer instance.
         */
        public static MessageComposer getInstance(String className, ConfigTree config, Class legacyAdapterType) throws ConfigurationException {
            MessageComposer instance;

            AssertArgument.isNotNullAndNotEmpty(className, "className");
            AssertArgument.isNotNull(config, "config");

            try {
                instance = newComposerInstance(ClassUtil.forName(className, Factory.class));
            } catch (ClassCastException e) {
                if (legacyAdapterType == null) {
                    throw new ConfigurationException("Composer class [" + className + "] not an instance of '" + MessageComposer.class.getName() + "' and no " + LegacyMessageComposerAdapter.class.getSimpleName() + " specified.", e);
                } else if (LegacyMessageComposerAdapter.class.isAssignableFrom(legacyAdapterType)) {
                    instance = newComposerInstance(legacyAdapterType);
                } else {
                    throw new ConfigurationException("Legacy composer adapter class [" + legacyAdapterType.getName() + "] not an instance of '" + LegacyMessageComposerAdapter.class.getName() + "'.");
                }
            } catch (ClassNotFoundException e) {
                throw new ConfigurationException("Composer class [" + className + "] not found in classpath.", e);
            }

            instance.setConfiguration(config);
            return instance;
        }

        private static MessageComposer newComposerInstance(Class composerClass) throws ConfigurationException {
            try {
                return (MessageComposer) composerClass.newInstance();
            } catch (InstantiationException e) {
                if(composerClass.getEnclosingClass() != null) {
                    throw new ConfigurationException("Failed to instantiate composer class [" + composerClass.getName() + "].  This may be because it is not a top level class.", e);
                } else {
                    throw new ConfigurationException("Failed to instantiate composer class [" + composerClass.getName() + "].", e);
                }
            } catch (IllegalAccessException e) {
                throw new ConfigurationException("Failed to instantiate composer class [" + composerClass.getName() + "].", e);
            }
        }
    }
}
