/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.listeners.jca;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.ListenerUtil;
import org.jboss.soa.esb.listeners.RegistryUtil;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.couriers.FaultMessageException;
import org.jboss.soa.esb.listeners.message.ActionProcessingPipeline;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.util.Util;

/**
 * @author <a href="kevin.conner@jboss.com">Kevin Conner</a>
 */
public class JcaMessageAwareListener extends BaseJcaInflow<InflowMessageProcessorAdapter> implements InflowMessageProcessor
{
    private final String serviceCategory ;
    private final String serviceName ;
    private final EPR serviceEPR ;
    private final JcaMessageAwareComposer composer ;
    
    private ActionProcessingPipeline pipeline ;

    public JcaMessageAwareListener(final ConfigTree config)
        throws ConfigurationException
    {
        super(config, InflowMessageProcessorAdapter.class);
        
        serviceCategory = ListenerUtil.getValue(config, ListenerTagNames.SERVICE_CATEGORY_NAME_TAG) ;
        if (Util.isNullString(serviceCategory))
        {
            throw new ConfigurationException("No service category defined!") ;
        }
        
        serviceName = ListenerUtil.getValue(config, ListenerTagNames.SERVICE_NAME_TAG) ;
        if (Util.isNullString(serviceName))
        {
            throw new ConfigurationException("No service name defined!") ;
        }
        
        final ConfigTree eprElement = config.getFirstChild(ListenerTagNames.EPR_TAG) ;
        if (eprElement == null)
        {
            throw new ConfigurationException("Missing EPR element") ;
        }
        
        serviceEPR = ListenerUtil.assembleEpr(eprElement) ;
        
        final String composerName = config.getAttribute(JcaConstants.ATTRIBUTE_LISTENER_COMPOSER_CLASS) ;
        if (Util.isNullString(composerName))
        {
            throw new ConfigurationException("No composer class defined") ;
        }
        
        final Class<?> composerClass ;
        try
        {
            composerClass = ClassUtil.forName(composerName, getClass()) ;
        }
        catch (final ClassNotFoundException cnfe)
        {
            throw new ConfigurationException("Could not load listener composer class", cnfe) ;
        }
        if (!JcaMessageAwareComposer.class.isAssignableFrom(composerClass))
        {
            throw new ConfigurationException("Listener composer class does not implement JcaMessageAwareComposer interface") ;
        }
        try
        {
            composer = (JcaMessageAwareComposer)composerClass.newInstance() ;
        }
        catch (final Throwable th)
        {
            throw new ConfigurationException("Unexpected exception caught while creating composer class", th) ;
        }
    }
    
    @Override
    protected String getDescription()
    {
        return "category: " + serviceCategory + " service: " + serviceName ;
    }
    
    @Override
    protected void doInitialise() throws ManagedLifecycleException
    {
        super.doInitialise();
        bean.setInflowMessageProcessor(this);
        
        try
        {
            pipeline = new ActionProcessingPipeline(getConfig()) ;
            pipeline.initialise() ;
        }
        catch (final ConfigurationException ce)
        {
            throw new ManagedLifecycleException("Unexpected exception configuring action processing pipeline", ce) ;
        }
        pipeline.setTransactional(isTransactional()) ;
        
        try
        {
            RegistryUtil.register(getConfig(), serviceEPR) ;
        }
        catch (final RegistryException re)
        {
            throw new ManagedLifecycleException("Unexpected exception while registering service", re) ;
        }
    }
    
    @Override
    protected void doDestroy() throws ManagedLifecycleException
    {
        super.doDestroy();
        
        try
        {
            pipeline.destroy() ;
        }
        catch (final Throwable ex)
        {
        }
        
        pipeline = null ;
        RegistryUtil.unregister(serviceCategory, serviceName, serviceEPR) ;
    }

    public void process(final Object messageIn)
    {
        try
        {
            try
            {
                final Message message = composer.compose(messageIn) ;
                if (message != null)
                {
                    pipeline.process(message) ;
                }
            }
            catch (final FaultMessageException fme)
            {
                pipeline.process(fme.getReturnedMessage()) ;
            }
        }
        catch (final RuntimeException re)
        {
            throw re ;
        }
        catch (final Error err)
        {
            throw err ;
        }
        catch (final Throwable th)
        {
            throw new JcaGatewayException("Unexpected exception caught while processing JCA message", th) ;
        }
    }
}
