/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.listeners.gateway;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import org.jboss.internal.soa.esb.listeners.gateway.filefilter.FileEndsWith;
import org.jboss.internal.soa.esb.listeners.gateway.filefilter.IgnoreFile;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.util.ClassUtil;
import org.jboss.soa.esb.util.FileUtil;

public class FileGatewayListener extends AbstractFileGateway {
    public interface FileFilterInit
    {
        void init (ConfigTree config) throws ConfigurationException;
    };
    
    /**
     * serial version uid for this class
     */
    private static final long serialVersionUID = 8457176329093772449L;

    public FileGatewayListener(ConfigTree config) throws ConfigurationException, RegistryException, GatewayException {
        super(config);

        _filterClass = config.getAttribute(ListenerTagNames.GATEWAY_FILE_FILTER_TAG);
        
        /*
           * JBESB-454 allowed gateways to pull files with any extension. Obvious
           * security issue, but we are explicit about this in the docs and users
           * should beware.
           */

        if (_filterClass == null)
        {
            if ((_inputSuffix == null) || (_inputSuffix.equals(""))) {
                /*
                    * If no suffix, then inputDir must be different from outputDir
                    * or we go into an infinite loop. Already checked through
                    * checkMyParams.
                    */
    
                _theFilter = new IgnoreFile(_workingSuffix, _errorSuffix, _postProcessSuffix);

                /*
                    * If no input suffix, then we must have a work suffix and we
                    * must ignore worker files when sourcing new input files, or
                    * we end up in an infinite loop.
                    */
            } else
                _theFilter = new FileEndsWith(_inputSuffix);
        }
        else
        {
            try
            {
                Class c = ClassUtil.forName(_filterClass, getClass());
                
                _theFilter = (FileFilter) c.newInstance();
                
                if (_theFilter instanceof FileFilterInit)
                {
                    ((FileFilterInit) _theFilter).init(config);
                }
            }
            catch (final ClassNotFoundException ex)
            {
                throw new ConfigurationException(ex);
            }
            catch (final IllegalAccessException ex)
            {
                throw new ConfigurationException(ex);
            }
            catch (final InstantiationException ex)
            {
                throw new ConfigurationException(ex);
            }
            catch (final ConfigurationException ex)
            {
                throw ex;
            }
            catch (final Exception ex)
            {
                throw new GatewayException(ex);
            }
        }
    }

    @Override
    protected void seeIfOkToWorkOnDir(File p_oDir) throws ConfigurationException {
        if (!p_oDir.exists())
            throw new ConfigurationException("Directory " + p_oDir.toString() + " not found");
        if (!p_oDir.isDirectory())
            throw new ConfigurationException(p_oDir.toString() + " is not a directory");
        if (!p_oDir.canRead())
            throw new ConfigurationException("Can't read directory " + p_oDir.toString());
        if (!p_oDir.canWrite())
            throw new ConfigurationException("Can't write/rename in directory " + p_oDir.toString());
    } // ________________________________

    @Override
    boolean deleteFile(File file) throws GatewayException {
        return file.delete();
    }

    @Override
    byte[] getFileContents(File file) throws GatewayException {
        try {
            ByteArrayOutputStream out = new ByteArrayOutputStream();
            byte[] ba = new byte[1000];  // TODO magic number
            int iQread;
            FileInputStream inp = new FileInputStream(file);
            while (-1 != (iQread = inp.read(ba)))
                if (iQread > 0)
                    out.write(ba, 0, iQread);
            inp.close();
            out.close();
            return out.toByteArray();
        }
        catch (IOException e) {
            throw new GatewayException(e);
        }
    }

    @Override
    File[] getFileList() throws GatewayException {
        /*
           * JBESB-454 allows no input suffix. This means that we need to
           * ignore any worker files, which are written to the same directory,
           * but which have a well defined suffix. All other files (errors and
           * post processed) go into separate directories, so we don't need
           * to be concerned about them.
           */

        return filterFiles(_inputDirectory, _theFilter);
    }

    private File[] filterFiles(File dir, FileFilter filter) {
        File[] files = dir.listFiles(filter);

        if(files == null) {
            _logger.warn("Error applying file filter '" + filter + "' to directory '" + dir.getAbsolutePath() + "'.  Null file list returned.  Invalid directory, or IO error reading directory.");
        }

        return files;
    }

    @Override
    boolean renameFile(File from, File to) throws GatewayException {
        return FileUtil.renameTo(from, to);
    }

    @Override
    String getDefaultComposer() {
         return LocalFileMessageComposer.class.getName();
    }

    @Override
    void bytesToFile(byte[] bytes, File file) throws GatewayException {
        try {
            if (file.exists() && !file.delete())
                throw new GatewayException("Unable to delete existing file " + file);
            FileOutputStream out = new FileOutputStream(file);
            out.write(bytes);
            out.close();
        }
        catch (Exception e) {
            throw new GatewayException(e);
        }
    }

    /**
     * This implementation allows no input suffix, whereas the base class
     * requires an input suffix.
     */

    protected void checkInputSuffix() throws ConfigurationException {
        // any suffix is allowed for input; set value to "" if null.

        if (_inputSuffix == null)
            _inputSuffix = "";
    }

    private FileFilter _theFilter;

    private String _filterClass = null;
}
