/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author JBoss Inc.
 */

package org.jboss.soa.esb.listeners.config.mappers120;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.Bus;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.DualListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.Listener;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.Provider;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.impl.BusImpl;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.impl.DualListenerImpl;
import org.jboss.soa.esb.listeners.config.xbeanmodel120.impl.BusProviderImpl;
import org.w3c.dom.Element;

/**
 * Perform a mapping for an "untyped" listener configuration.
 * <p/>
 * By "untyped" we mean that the listener type is not defined in the XSD, but instead is being
 * defined using the base &lt;listener&gt; type, mapping all it's associated
 * &lt;property&gt; values (including those of the associated bus and provider) onto the listener
 * ConfigTree as attributes.
 * <p/>
 * This is effectively like the base transform - if we don't have a mapper for the listener, this one is applied.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class UntypedListenerMapper {

	/**
	 * Perform the mapping.
	 * @param root The "ConfigTree" configuration root node.
	 * @param listener The listener to be mapped into the ConfigTree.
	 * @param model The configuration model from which the mapping is being performed.
	 * @return The ConfigTree listener configuration node.
	 * @throws org.jboss.soa.esb.ConfigurationException Invalid listener configuration.
	 */
	public static Element map(Element root, DualListener listener, XMLBeansModel model) throws ConfigurationException {
		Element listenerNode = YADOMUtil.addElement(root, "listener");

        listenerNode.setAttribute("name", listener.getName());

		assertListenerConfigOK(listener, model);

		// Map the standard listener attributes - common across all listener types...
		MapperUtil.mapDefaultAttributes(listener, listenerNode, model);
		// Map the <property> elements targeted at the listener - from the listener itself.
		MapperUtil.mapProperties(listener.getPropertyList(), listenerNode);

		if(listener.getIsGateway()) {
			listenerNode.setAttribute(ListenerTagNames.IS_GATEWAY_TAG, Boolean.toString(listener.getIsGateway()));
			
			MapperUtil.mapEPRProperties(listener, listenerNode, model);
		} else {
			Element eprNode = YADOMUtil.addElement(listenerNode, ListenerTagNames.EPR_TAG);
			MapperUtil.mapEPRProperties(listener, eprNode, model);

			// Remove any empty attributes set on the EPR config...
			YADOMUtil.removeEmptyAttributes(eprNode);
		}

		// Remove any empty attributes set on the listener config...
		YADOMUtil.removeEmptyAttributes(listenerNode);

		return listenerNode;
	}

	/**
	 * Assert that a valid mapping can be performed from the supplied listener, and the bus + provider it reverences.
	 * @param listener The listener to be checked.
	 * @param model The config model instance.
	 * @throws org.jboss.soa.esb.ConfigurationException Invalid mappng request.
	 */

	private static void assertListenerConfigOK(Listener listener, XMLBeansModel model) throws ConfigurationException {

		// Note we're purposely performing instanceof comparisons here...

		if(listener.getClass() != DualListenerImpl.class) {
			throw new ConfigurationException("Can only use the " + UntypedListenerMapper.class.getName() + " mapper on the base listener type.  Cannot use on " + listener.getClass().getName());
		}
		Bus bus = model.getBus(listener.getBusidref());
		if(bus.getClass() != BusImpl.class) {
			throw new ConfigurationException("The base Listener config [" + listener.getName() + "] must reference a base Bus config type (<bus>).");
		}
		Provider provider = model.getProvider(bus);
		if(provider.getClass().isAssignableFrom(BusProviderImpl.class)) {
			throw new ConfigurationException("A base Bus config type (<bus>) must be contained within a base Provider type (<bus-provider>).");
		}
	}
}