/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.config.mappers110;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.JbrBusDocument.JbrBus;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.JbrListenerDocument.JbrListener;
import org.jboss.soa.esb.listeners.config.xbeanmodel110.JbrProviderDocument.JbrProvider;
import org.jboss.soa.esb.listeners.gateway.JBossRemotingGatewayListener;
import org.w3c.dom.Element;

/**
 * JBoss Remoting listener mapper.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class JbrListenerMapper {

    /**
     * Perform the mapping.
     * @param root The "ConfigTree" configuration root node.
     * @param listener JBoss Remoting listener config.
     * @param model The configuration model from which the mapping is being performed.
     * @return The ConfigTree listener configuration node.
     * @throws org.jboss.soa.esb.ConfigurationException Invalid listener configuration.
     */
    public static Element map(Element root, JbrListener listener, XMLBeansModel model) throws ConfigurationException {
        Element listenerNode = YADOMUtil.addElement(root, "listener");
        JbrBus bus;
        JbrProvider provider;

        listenerNode.setAttribute("name", listener.getName());

        try {
            bus = (JbrBus) model.getBus(listener.getBusidref());
        } catch (ClassCastException e) {
            throw new ConfigurationException("Invalid busid reference [" + listener.getBusidref() + "] on listener [" + listener.getName() + "].  A <jbr-listener> must reference a <jbr-bus>.");
        }
        try {
            provider = (JbrProvider) model.getProvider(bus);
        } catch (ClassCastException e) {
            throw new ConfigurationException("Invalid bus config [" + listener.getBusidref() + "].  Should be contained within a <jbr-provider> instance.  Unexpected exception - this should have caused a validation error!");
        }

        // Map the standard listener attributes - common across all listener types...
        MapperUtil.mapDefaultAttributes(listener, listenerNode, model);
        // Map the <property> elements targeted at the listener - from the listener itself.
        MapperUtil.mapProperties(provider.getPropertyList(), listenerNode);
        MapperUtil.mapProperties(bus.getPropertyList(), listenerNode);
        MapperUtil.mapProperties(listener.getPropertyList(), listenerNode);

        if(listener.getIsGateway()) {
            String host = provider.getHost();

            listenerNode.setAttribute("gatewayClass", JBossRemotingGatewayListener.class.getName());
			listenerNode.setAttribute(ListenerTagNames.IS_GATEWAY_TAG, Boolean.toString(listener.getIsGateway()));

            if(host != null) {
                listenerNode.setAttribute(JBossRemotingGatewayListener.JBR_SERVER_HOST, host);
            }
            listenerNode.setAttribute(JBossRemotingGatewayListener.JBR_SERVER_PROTOCOL, provider.getProtocol().toString());
            listenerNode.setAttribute(JBossRemotingGatewayListener.JBR_SERVER_PORT, Integer.toString(bus.getPort()));
        } else {
            throw new ConfigurationException("Invalid <jbr-listener> config [" + listener.getName() + "].  <jbr-listener> is currently only supported as a gateway listener.");
        }

        return listenerNode;
    }
}
