/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.http;

import org.jboss.soa.esb.services.security.auth.AuthenticationRequest;
import org.jboss.soa.esb.services.security.auth.AuthenticationRequestImpl;
import org.jboss.soa.esb.services.security.PublicCryptoUtil;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.util.Base64;
import org.apache.log4j.Logger;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.ServletException;
import java.util.Set;
import java.util.HashSet;
import java.security.Principal;

/**
 * Http Servlet security utilities.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class HttpServletSecUtil {

    private static final Logger logger = Logger.getLogger(HttpServletSecUtil.class);

    public static boolean addAuthDetailsToMessage(HttpServletRequest request, Message message) throws ServletException {
        Principal userPrincipal = request.getUserPrincipal();

        if(userPrincipal != null) {
            // It's a logged in user.  Need to get the login details from the Authorization header
            // and pass it along in the ESB Message sent to the Service...
            AuthenticationRequest authRequest = buildAuthenticationRequest(request);

            // Set the AuthenticationRequest on the inbound message...
            try {
                return PublicCryptoUtil.INSTANCE.addAuthRequestToMessage(authRequest, message);
            } catch (MessageDeliverException e) {
                throw new ServletException("Failed to attach AuthenticationRequest to ESB Message.", e);
            }
        }

        return false;
    }

    public static AuthenticationRequest buildAuthenticationRequest(HttpServletRequest req) {
        String authHeader = req.getHeader("Authorization");

        if(authHeader == null) {
            authHeader = req.getHeader("authorization");
        }

        if(authHeader != null) {
            // Handle BASIC auth...
            if("BASIC".equals(req.getAuthType())) {
                String bas64Data = authHeader.substring(authHeader.indexOf(' ')).trim();
                String decodedData = new String(Base64.decode(bas64Data));
                String[] authTokens = decodedData.split(":");
                Set<Object> credentials = new HashSet<Object>();

                // The auth tokens are "username:password", so the second token is the
                // password, which is the credential in this case...
                credentials.add(authTokens[1].toCharArray());

                if(logger.isDebugEnabled()) {
                    logger.debug("User '" + authTokens[0] + "' authenticated by container using 'BASIC' auth.  Forwarding to ESB Service.");
                }

                return new AuthenticationRequestImpl.Builder(req.getUserPrincipal(), credentials).build();
            } else {
                if(logger.isDebugEnabled()) {
                    logger.debug("Authentication  method '" + req.getAuthType() + "' not supported for passing Authorization token to ESB Security.");
                }
            }
            // TODO: Add support for Client-cert and Digest auth?
        }

        return null;
    }
}
