package org.jboss.soa.esb.addressing.eprs;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/**
 * This class represents the endpoint reference for services.
 */

import java.net.URI;
import java.net.URISyntaxException;

import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.addressing.XMLUtil;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

/**
 * A helper class for using SFTP style EPRs.
 * 
 * @author marklittle
 */
public class SFTPEpr extends FTPEpr
{
        public static final String CERTIFICATE_TAG = "certificate";
        public static final String PASSPHRASE_TAG = "passphrase" ;

        public SFTPEpr(EPR epr)
        {
                super(epr);
                
                if (epr instanceof SFTPEpr)
                {
                    certificateSet = ((SFTPEpr) epr).certificateSet;
                }
        }
        
        public SFTPEpr (EPR epr, Element header)
        {
                super(epr, header);
                
                NodeList nl = header.getChildNodes();

                for (int i = 0; i < nl.getLength(); i++)
                {
                        String prefix = nl.item(i).getPrefix();
                        String tag = nl.item(i).getLocalName();
                        
                        try
                        {
                                if ((prefix != null) && (prefix.equals(XMLUtil.JBOSSESB_PREFIX)))
                                {
                                        if (tag != null)
                                        {
                                                if(tag.equals(CERTIFICATE_TAG))
                                                {
                                                        getAddr().addExtension(CERTIFICATE_TAG, nl.item(i).getTextContent());
                                                        certificateSet = true;
                                                }
                                                else if (tag.equals(PASSPHRASE_TAG))
                                                {
                                                        getAddr().addExtension(PASSPHRASE_TAG, nl.item(i).getTextContent()) ;
                                                }
                                        }
                                }
                        }
                        catch (Exception ex)
                        {
                                ex.printStackTrace();
                        }
                }
        }

       /**
         * Construct the SFTP EPR with the specified URI.
         * @param uri
         * @throws URISyntaxException
         */
        public SFTPEpr(URI uri)
        {
                super(uri);
        }

        /**
         * Construct the SFTP EPR with the specified URI and certificate.
         * @param uri
         * @param cert
         * @throws URISyntaxException
         */
        public SFTPEpr(URI uri, URI cert) throws URISyntaxException
        {
            super(uri) ;
            
            setCertificateURI(cert);
        }

        /**
         * Construct the SFTP EPR with the specified URI, certificate and passphrase.
         * @param uri
         * @param cert
         * @param passphrase
         * @throws URISyntaxException
         */
        public SFTPEpr(URI uri, URI cert, final String passphrase) throws URISyntaxException
        {
                super(uri) ;
                
                setCertificateURI(cert);
                setPassphrase(passphrase) ;
        }

        public SFTPEpr(String url, String cert) throws URISyntaxException
        {
                super(url);     

                setCertificateURI(new URI(cert));
        }

        public SFTPEpr(String url, String cert, final String passphrase) throws URISyntaxException
        {
                super(url);

                setCertificateURI(new URI(cert));
                setPassphrase(passphrase) ;
        }

        /**
         * Set the Certificate URI for this endpoint.
         * 
         * @param cert
         *            the address.
         * @throws URISyntaxException
         *             thrown if this EPR is malformed.
         */
        public final void setCertificateURI(URI cert) throws URISyntaxException
        {
                if (cert == null)
                        throw new IllegalArgumentException();

                if (certificateSet)
                        throw new IllegalStateException("Certificate URI already set.");

                getAddr().addExtension(CERTIFICATE_TAG, cert.toASCIIString());
                certificateSet = true;
        }

        /**
         * @return the certificate URI for this EPR.
         */

        public final URI getCertificateURI () throws URISyntaxException
        {
            try
            {
                String cert = getAddr().getExtensionValue(CERTIFICATE_TAG);

                if (cert != null)
                        return new URI(cert);
                else
                        return null;
            }
            catch (Exception ex)
            {
                _logger.warn("Unexpected parsing exception!", ex);
                
                return null;
            }
        }

        /**
         * Set the Certificate passphrase for this endpoint.
         * 
         * @param passphrase the passphrase.
         */
        public final void setPassphrase(String passphrase)
        {
                if (passphrase == null)
                        throw new IllegalArgumentException();
                getAddr().addExtension(PASSPHRASE_TAG, passphrase);
        }

        /**
         * @return the certificate passphrase for this EPR.
         */
        public final String getPassphrase()
        {
                return getAddr().getExtensionValue(PASSPHRASE_TAG);
        }
        
        public EPR copy ()
        {
            return new SFTPEpr(this);
        }
        
        public String toString ()
        {
                return "SFTPEpr [ "+super.getAddr().extendedToString()+" ]";
        }
        
        public static URI type ()
        {
                return _type;
        }

        private boolean certificateSet = false;

        private static URI _type;

        static
        {
                try
                {
                        _type = new URI("urn:jboss/esb/epr/type/sftp");
                }
                catch (Exception ex)
                {
                        ex.printStackTrace();

                        throw new ExceptionInInitializerError(ex.toString());
                }
        }
}
