package org.jboss.soa.esb.actions.validation;

import java.io.IOException;
import java.io.StringReader;

import javax.xml.XMLConstants;
import javax.xml.transform.Source;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.util.ClassUtil;
import org.xml.sax.SAXException;

/**
 * An action that validates the passed-in message payload against the configured schema.
 * The payload can be of either type byte[] or String.
 *
 * Example configuration:
 * <pre>{@code
 * <action name="validate" class="org.jboss.soa.esb.actions.validation.SchemaValidationAction">
 *     <property name="schema" value="/test.xsd"/>
 * </action>
 * }</pre>
 * Optional properties:
 * <lu>
 * <li><i>schemaLanguage</i> the schema langauage to use. Defaults to "http://www.w3.org/2001/XMLSchema".</li>
 * </lu>
 *
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 *
 */
public class SchemaValidationAction extends AbstractActionPipelineProcessor
{
    /**
     * Logger.
     */
    final Logger log = Logger.getLogger(SchemaValidationAction.class);

    /**
     * Path to the xsd.
     */
    private String xsd;

    /**
     * Schema language.
     */
    private String schemaLanguage;

    /**
     * The schema that will be used for validation.
     */
    private Schema schema;

    /**
     * The {@link MessagePayloadProxy}.
     */
    private MessagePayloadProxy payloadProxy;

    /**
     * Sole constructor that expects the config tree to have the attribute "schema"
     *
     * @param config    The action configuration.
     * @throws ConfigurationException
     */
    public SchemaValidationAction(final ConfigTree config) throws ConfigurationException
    {
        xsd = config.getRequiredAttribute("schema");
        schemaLanguage = config.getAttribute("schemaLanguage", XMLConstants.W3C_XML_SCHEMA_NS_URI);

        payloadProxy = new MessagePayloadProxy(config);
        schema = createSchema(xsd);
    }

    /**
     * Processes the messsage, extracting the payload and validating the xml against the
     * configured schema.
     *
     * @param message   The esb Message object.
     * @return Message  The esb Message unmodified.
     *
     * @throws ActionProcessingException If a exception occurs while trying to validate
     *
     */
    public Message process(final Message message) throws ActionProcessingException
    {
        validate(extractXmlFromPayload(message));
        return message;
    }

    private void validate(final String xml) throws ActionProcessingException
    {
        try
        {
            final Validator validator = schema.newValidator();
            validator.validate(createSourceFromPayload(xml));
        }
        catch (final SAXException e)
        {
            final String errorMsg = "SAXException while trying to validate against schema '" + xsd + "'";
            log.error(errorMsg, e);
            throw new ValidationException(errorMsg, e);
        }
        catch (IOException e)
        {
            throw new ActionProcessingException(e.getMessage(), e);
        }
    }

    private String extractXmlFromPayload(final Message message) throws ActionProcessingException
    {
        Object payload;
        try
        {
            payload = payloadProxy.getPayload(message);
        }
        catch (final MessageDeliverException e)
        {
            throw new ActionProcessingException(e.getMessage(), e);
        }

        if (payload instanceof byte[])
        {
            return new String((byte[]) payload);
        }
        else if (payload instanceof String)
        {
            return (String) payload;
        }
        else
        {
            throw new ActionProcessingException("Message payload must be either a byte[] or a String. The payload type was '" + payload.getClass().getName() + "'");
        }
    }

    /**
     * Returns a DOMSource from the passed in xml String.
     *
     * @param xml           The xml payload.
     * @return DOMSource    The {@link DOMSource} representation of the passed in xml.
     *
     * @throws ActionProcessingException If an exception occurs while trying to create the DOMSource.
     */
    private Source createSourceFromPayload(final String xml) throws ActionProcessingException
    {
    	return new StreamSource(new StringReader(xml));
    }

    /**
     * Creates a Validator instance which will be used for schema validation.
     *
     * @param xsd           The schema to validate against.
     * @return Validator    The Validator instance.
     *
     * @throws ConfigurationException Is an exception is thrown while trying to create the schema.
     */
    private Schema createSchema(final String xsd) throws ConfigurationException
    {
        final SchemaFactory schemafactory = SchemaFactory.newInstance(schemaLanguage);
        try
        {
            return schemafactory.newSchema(new StreamSource(ClassUtil.getResourceAsStream(xsd, getClass())));
        }
        catch (final SAXException e)
        {
            throw new ConfigurationException("Could not create a validator for schema '" + xsd + "'", e);
        }
    }
}
