/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.transformation.xslt;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import javax.xml.transform.URIResolver;

import org.jboss.soa.esb.actions.transformation.xslt.ResultFactory.ResultType;

/**
 * Simple data holder for Tranformer factory configuration values.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 * @since 4.6
 */
public class TransformerFactoryConfig
{
    /**
     * The xslt template file path.
     */
    private final String templateFile;
    
    /**
     * The xslt template file path.
     */
    private final ResultType resultType;
    
    /**
     * Map for transformer factory features.
     */
    private final Map<String, Boolean> features;
    
    /**
     * Map of transformer factory attributes.
     */
    private final Map<String, Object> attributes;
    
    /**
     * {@link URIResolver} to be used.
     */
    private URIResolver uriResolver;
    
    private TransformerFactoryConfig(final Builder builder)
    {
        templateFile = builder.templateFile;
        features = builder.features;
        attributes = builder.attributes;
        uriResolver = builder.uriResolver;
        resultType = builder.resultType;
    }
    
    public String getTemplateFile()
    {
        return templateFile;
    }
    
    public Map<String, Boolean> getFeatures()
    {
        return features;
    }
    
    public Map<String, Object> getAttributes()
    {
        return attributes;
    }
    
    public URIResolver getUriResolver()
    {
        return uriResolver;
    }
    
    public ResultType getResultType()
    {
        return resultType;
    }
    
    public static class Builder
    {
        private final String templateFile;
        private Map<String, Boolean> features;
        private Map<String, Object> attributes;
        private URIResolver uriResolver;
        private ResultType resultType;

        public Builder(final String templateFile)
        {
            this.templateFile = templateFile;
        }
        
        public Builder feature(final String name, final Boolean value)
        {
            if (features == null)
                features = new HashMap<String, Boolean>();
            
            features.put(name, value);
            return this;
        }
        
        public Builder attribute(final String name, final Object value)
        {
            if (attributes == null)
                attributes = new HashMap<String, Object>();
            
            attributes.put(name, value);
            return this;
        }
        
        public Builder uriResolver(final URIResolver resolver)
        {
            uriResolver = resolver;
            return this;
        }
        
        public Builder resultType(final ResultType type)
        {
            resultType = type;
            return this;
        }
        
        public TransformerFactoryConfig build()
        {
            if (features == null)
                features = Collections.emptyMap();
            
            if (attributes == null)
                attributes = Collections.emptyMap();
            
            if (resultType == null)
                resultType = ResultType.STRING;
            
            return new TransformerFactoryConfig(this);
        }
    }
}
