/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.transformation.xslt;

import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.io.Reader;
import java.io.StringReader;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.transform.Source;
import javax.xml.transform.sax.SAXSource;
import javax.xml.transform.stream.StreamSource;

import org.xml.sax.ErrorHandler;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

/**
 * Code contributed from the Smooks project.
 */
public class SourceFactory 
{
    private static final SourceFactory factory = new SourceFactory();

    private SourceFactory() {}

    public static SourceFactory getInstance() 
    {
        return factory;
    }
    
    public Source createSource(final Object from)
	{
    	try
    	{
    		return createSource(from, null, null);
    	}
    	catch (SAXException e)
    	{
    		throw new RuntimeException("Should not occur when not validating XML InputSource!", e);
    	}
    	catch (ParserConfigurationException e)
    	{
    		throw new RuntimeException("Should not occur when not validating XML InputSource!", e);
    	}
	}

    public Source createSource(final Object from, final SAXParserFactory validationFactory, final ErrorHandler validationHandler)
    	throws SAXException, ParserConfigurationException 
    {
        final Source source;
        final XMLReader validationReader;
        if (validationFactory != null)
        {
        	validationReader = validationFactory.newSAXParser().getXMLReader();
        	if (validationHandler != null)
        	{
        		validationReader.setErrorHandler(validationHandler);
        	}
        }
        else
        {
        	validationReader = null;
        }
        if (from instanceof String) 
        {
        	Reader reader = new StringReader((String)from);
        	if (validationReader != null)
        	{
        		source = new SAXSource(validationReader, new InputSource(reader));
        	}
        	else
        	{
        		source = new StreamSource(reader);
        	}
        } 
        else if (from instanceof byte[]) 
        {
        	InputStream inputStream = new ByteArrayInputStream((byte[])from);
        	if (validationReader != null)
        	{
        		source = new SAXSource(validationReader, new InputSource(inputStream));
        	}
        	else
        	{
        		source = new StreamSource(inputStream);
        	}
        } 
        else if (from instanceof Reader) 
        {
        	Reader reader = (Reader)from;
        	if (validationReader != null)
        	{
        		source = new SAXSource(validationReader, new InputSource(reader));
        	}
        	else
        	{
        		source = new StreamSource(reader);
        	}
        } 
        else if (from instanceof InputStream) 
        {
        	InputStream inputStream = (InputStream)from;
        	if (validationReader != null)
        	{
        		source = new SAXSource(validationReader, new InputSource(inputStream));
        	}
        	else
        	{
        		source = new StreamSource(inputStream);
        	}
        } 
        else if (from instanceof File)
        {
        	InputStream inputStream;
            try
            {
            	inputStream = new BufferedInputStream(new FileInputStream((File)from));
            } 
            catch (FileNotFoundException e)
            {
                throw new IllegalStateException(e.getMessage(), e);
            }
        	if (validationReader != null)
        	{
        		source = new SAXSource(validationReader, new InputSource(inputStream));
        	}
        	else
        	{
        		source = new StreamSource(inputStream);
        	}
        }
        else if (from instanceof Source) 
        {
            source = (Source)from;
        } 
        else 
        {
            throw new IllegalStateException("Object '" + from + "' is not of a supported type (String, byte[], Reader, InputStream, File, or Source). Try using a SourceResult instead perhaps.");
        }
        return source;
    }
}
