/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions.naming;

import java.io.IOException;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;

import org.apache.xerces.parsers.DOMParser;
import org.jboss.soa.esb.actions.AbstractActionPipelineProcessor;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.actions.BeanConfiguredAction;
import org.jboss.soa.esb.actions.naming.strategy.FileNamingStrategy;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.util.ClassUtil;
import org.w3c.dom.DOMException;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Action that generates a filename based on a naming strategy.
 * 
 * @author Derek Adams
 */
public class FileNameGeneratorAction extends AbstractActionPipelineProcessor implements
		BeanConfiguredAction {

	/** Property name for incoming file name */
	private String fileNameProperty = Environment.ORIGINAL_FILE_NAME_MSG_PROP;

	/** Property name for filename after processing */
	private String resultProperty = DEFAULT_RESULT_PROPERTY;

	/** String containing XML for configured strategies */
	private String strategies;

	/** List of strategy instances created from XML */
	protected List<FileNamingStrategy> strategyImpls;

	/** Default name for processing result */
	public static final String DEFAULT_RESULT_PROPERTY = "org.jboss.soa.esb.naming.result";

	/** Tag for strategy element */
	public static final String ELEMENT_STRATEGY = "strategy";
	
	/** Class attribute */
	public static final String ATTR_CLASS = "class";

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.actions.ActionPipelineProcessor#process(org.jboss.soa.esb.message.Message)
	 */
	public Message process(Message message) throws ActionProcessingException {
		String currentFilename = (String) message.getProperties().getProperty(getFileNameProperty());
		for (FileNamingStrategy strategy : strategyImpls) {
			currentFilename = strategy.process(currentFilename, message);
		}
		message.getProperties().setProperty(getResultProperty(), currentFilename);
		return message;
	}

	/**
	 * Create a List of strategy implentations from the XML.
	 */
	protected void createStrategies() {
		strategyImpls = new ArrayList<FileNamingStrategy>();
		try {
			DOMParser parser = new DOMParser();
			InputSource source = new InputSource(new StringReader(getStrategies()));
			parser.parse(source);
			Document doc = parser.getDocument();
			NodeList strategyNodes = doc.getElementsByTagName(ELEMENT_STRATEGY);
			for (int i = 0; i < strategyNodes.getLength(); i++) {
				FileNamingStrategy strategy = createStrategy(strategyNodes.item(i));
				strategyImpls.add(strategy);
			}
		} catch (SAXException e) {
			throw new RuntimeException(e);
		} catch (IOException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Create a strategy instance given a DOM node.
	 * 
	 * @param strategyNode
	 * @return FileNamingStrategy
	 */
	protected FileNamingStrategy createStrategy(Node strategyNode) {
		Node classNode = strategyNode.getAttributes().getNamedItem(ATTR_CLASS);
		if (classNode == null) {
			throw new RuntimeException("Strategy does not have 'class' attribute.");
		}
		try {
			Class strategyClass = ClassUtil.forName(classNode.getNodeValue(), this.getClass());
			FileNamingStrategy strategy = (FileNamingStrategy) strategyClass.newInstance();
			strategy.configure(strategyNode);
			return strategy;
		} catch (DOMException e) {
			throw new RuntimeException(e);
		} catch (ClassNotFoundException e) {
			throw new RuntimeException(e);
		} catch (InstantiationException e) {
			throw new RuntimeException(e);
		} catch (IllegalAccessException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * Set the property that holds the initial file name.
	 * 
	 * @param fileNameProperty
	 */
	public void setFileNameProperty(String fileNameProperty) {
		this.fileNameProperty = fileNameProperty;
	}

	/**
	 * Get the property that holds the initial file name.
	 * 
	 * @return String
	 */
	public String getFileNameProperty() {
		return fileNameProperty;
	}

	/**
	 * Set the property that holds the processed file name.
	 * 
	 * @param resultProperty
	 */
	public void setResultProperty(String resultProperty) {
		this.resultProperty = resultProperty;
	}

	/**
	 * Get the property that holds the processed file name.
	 * 
	 * @return String
	 */
	public String getResultProperty() {
		return resultProperty;
	}

	/**
	 * Set the string containing XML for strategies.
	 * 
	 * @param strategies
	 */
	public void setStrategies(String strategies) {
		this.strategies = strategies;
		this.createStrategies();
	}

	/**
	 * Set the string containing XML for strategies.
	 * 
	 * @return String
	 */
	public String getStrategies() {
		return strategies;
	}
}