/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006
 */
package org.jboss.soa.esb.actions;

import java.io.PrintStream;

import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.util.Util;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;

/**
 * Simple action that prints out the message contents using System.println.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class SystemPrintln extends AbstractActionPipelineProcessor
{
	public static final String PRE_MESSAGE = "message";
	public static final String FULL_MESSAGE = "printfull";
	public static final String PRINT_STREAM = "outputstream";
    public static final String DEFAULT_PRE_MESSAGE = "Message structure";
    
    private MessagePayloadProxy payloadProxy;

    /**
	 * Public constructor.
	 * 
	 * If no PRE_MESSAGE data is provided within the supplied ConfigTree instance
	 * then DEFAULT_PRE_MESSAGE is used.
	 * 
	 * @param config
	 *            Configuration.
	 */
	public SystemPrintln(ConfigTree config)
	{
		printlnMessage = config.getAttribute(PRE_MESSAGE, DEFAULT_PRE_MESSAGE);
		printFullMessage = (config.getAttribute(FULL_MESSAGE, "false").equalsIgnoreCase("true") ? true : false);
		useOutputStream = (config.getAttribute(PRINT_STREAM, "true").equals("true") ? true : false);

        String primaryDataLocation = config.getAttribute("datalocation");
        if(primaryDataLocation != null) {
            config.setAttribute(MessagePayloadProxy.GET_PAYLOAD_LOCATION, primaryDataLocation);
            payloadProxy = new MessagePayloadProxy(config);
        } else {
            payloadProxy = new MessagePayloadProxy(config,
                                                   new String[] {BytesBody.BYTES_LOCATION},
                                                   new String[] {BytesBody.BYTES_LOCATION});
        }
        payloadProxy.setNullGetPayloadHandling(MessagePayloadProxy.NullPayloadHandling.LOG);
    }

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jboss.soa.esb.actions.ActionProcessor#process(org.jboss.soa.esb.message.Message)
	 */
	public Message process(Message message) throws ActionProcessingException
	{
	    Object messageObject = null;
	    try {
	        messageObject = payloadProxy.getPayload(message);
	    } catch (MessageDeliverException e) {
	        throw new ActionProcessingException(e);
	    }

	    PrintStream stream = (useOutputStream ? System.out : System.err);

	    stream.println(printlnMessage + ": ");

	    String messageStr=null;

	    if (printFullMessage && (message != null))
	    {
	        // the message should be responsible for converting itself to a string
	        messageStr = message.toString();
	        stream.println("[ "+messageStr+" ]");

	    }
	    else
	    {
	        if (messageObject instanceof byte[])
	        {
	            messageStr = Util.format(new String((byte[]) messageObject));
	            stream.println("[" + messageStr + "].");
	        }
	        else
	        {
	            if (messageObject != null) {
	                messageStr = Util.format(messageObject.toString());
	                stream.println("[" + messageStr + "].");
	            }
	            
	            for (int i = 0; i < message.getAttachment().getUnnamedCount(); i++)
	            {
	                Object payload = message.getAttachment().itemAt(i);

	                /*
	                 * If the attachment is a message then get the contents from it.
	                 * Otherwise treat the attachment as the payload we want to deal
	                 * with.
	                 */
	                
	                if (payload instanceof Message)
	                {
	                    Message attachedMessage = (Message) payload;
	                    
	                    try {
	                        payload = payloadProxy.getPayload(attachedMessage);
	                    } catch (MessageDeliverException e) {
	                        throw new ActionProcessingException(e);
	                    }
	                }

	                    if(payload instanceof byte[]) {
	                        stream.println("attachment " + i + ": ["
	                                + new String((byte[]) payload)
	                        + "].");
	                    } else {
	                        stream.println("attachment " + i + ": ["
	                                + payload
	                                + "].");
	                    }
	                }
	                }
	            }

	        return message;
	}

	private String printlnMessage;
	private boolean printFullMessage;
	private boolean useOutputStream;
}
