/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.body.content.BytesBody;
import org.jboss.soa.esb.notification.NotificationList;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;

import java.util.ArrayList;
import java.util.List;

/**
 * Stand alone action class that replaces the NotificationHandler EJB (beta 1)
 *
 * @author <a href="mailto:schifest@heuristica.com.ar">schifest@heuristica.com.ar</a>
 * @since Version 4.0
 */
public class Notifier implements ActionLifecycle {

    private MessagePayloadProxy payloadProxy;

    public Notifier(ConfigTree config) {
        _config = config;
        payloadProxy = new MessagePayloadProxy(config,
                                               new String[] {BytesBody.BYTES_LOCATION},
                                               new String[] {BytesBody.BYTES_LOCATION});
    } //________________________________

    /**
     * Initialise the action instance.
     * <p/>
     * This method is called after the action instance has been instantiated so that
     * configuration options can be validated.
     *
     * @throws ActionLifecycleException for errors during initialisation.
     */
    public void initialise()
            throws ActionLifecycleException {
        if (null == _config) {
            _notifyOK = _notifyError = new ConfigTree[]{};
        } else {
            _notifyOK = getNotificationList("ok");
            _notifyError = getNotificationList("err");
        }
    }

    /**
     * Destroy the action instance.
     * <p/>
     * This method is called prior to the release of the action instance.  All
     * resources associated with this action instance should be released as the
     * instance will no longer be used.
     */
    public void destroy()
            throws ActionLifecycleException {
    }

    /**
     * This is equivalent to a No Operation
     *
     * @param message
     * @return - this method will always return arg0 unchanged
     */
    public Message process(Message message) {
        return message;
    } //________________________________

    public void notifyOK(Message message) {
        NotificationList.notifyAll(_notifyOK, message);
    } //________________________________

    public void notifyError(Message message, final Throwable th) {
        NotificationList.notifyAll(_notifyError, message);
    } //________________________________

    @Deprecated
    public String messageAsString(Message message) throws MessageDeliverException {
        byte[] ba = null;
        return (null == message) ? "<null message>"
                : (null == (ba = (byte[]) payloadProxy.getPayload(message))) ? "<null body content>"
                : new String(ba);
    } //________________________________

    private ConfigTree[] getNotificationList(String type) {
        List<ConfigTree> list = new ArrayList<ConfigTree>();
        if (null != type)
            type = type.toLowerCase();

        for (ConfigTree tree : _config.getChildren(NotificationList.ELEMENT)) {
            String sType = tree.getAttribute(NotificationList.TYPE);
            if (null == sType)
                continue;
            if (null == type || sType.toLowerCase().equals(type))
                list.add(tree);
        }

        ConfigTree[] array = new ConfigTree[list.size()];
        return list.toArray(array);
    } //________________________________

    protected ConfigTree _config;
    protected ConfigTree[] _notifyOK, _notifyError;
    protected static Logger _logger = Logger.getLogger(Notifier.class);
} //____________________________________________________________________________
