/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.webservice;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.AccessController;
import java.security.PrivilegedAction;

import org.apache.log4j.Logger;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.listeners.config.WebserviceInfo;


public class ESBServiceEndpointInfo {
	private static final String DEFAULT_ENCODING ;
	private static final Logger LOGGER = Logger.getLogger(ESBServiceEndpointInfo.class) ;

	private final String requestName ;
	private final String responseName ;
	private final String faultName ;
	private final String bindingName ;
	private final String serviceName ;
	private final String portName ;
	private final String operationName ;
	private final String servletName ;
	private final String servletPath ;
	private final String wsdlFileName ;
	private final String packageName ;
	private final String className ;
	private final String namespace ;
	private final boolean isOneWay ;
	private final String requestAction ;
	private final String responseAction ;
	private final String requestLocation ;
	private final String responseLocation ;
	private final boolean addressing ;
	
	public ESBServiceEndpointInfo(final WebserviceInfo webserviceInfo)
			throws UnsupportedEncodingException {
		final Service service = webserviceInfo.getService() ;
		final String name = service.getName() ;
		requestName = name + "Req" ;
		responseName = name + "Res" ;
		faultName = name + "Fault" ;
		bindingName = name + "Binding" ;
		serviceName = name + "Service" ;
		portName = name + "PortType" ;
		operationName = name + "Op" ;
		final String serviceCategory = service.getCategory().replace('/', '_') ;
		final String serviceName = service.getName().replace('/', '_') ;
		servletName = serviceCategory + "_" + serviceName ;
		servletPath = "/" + serviceCategory + "/" + serviceName ;
		wsdlFileName = "WEB-INF/wsdl/" + serviceCategory + "/" + serviceName + ".wsdl" ;
		packageName = "esb.ws." + serviceCategory + "." + serviceName ;
		className = packageName + ".Implementation" ;
		namespace = "http://soa.jboss.org/" + URLEncoder.encode(service.getCategory(), DEFAULT_ENCODING) ;
		requestAction = namespace + "/" + operationName ;
		responseAction = requestAction + "Resp" ;
		
		isOneWay = (webserviceInfo.getOutXsd() == null) ;
		if (isOneWay)
		{
			if (webserviceInfo.getFaultXsd() != null)
			{
				LOGGER.warn("Fault schemas ignored for OneWay service " + serviceCategory + ":" + serviceName) ;
			}
			if (webserviceInfo.isRequestResponse())
			{
				LOGGER.warn("No response specified for RequestResponse service " + serviceCategory + ":" + serviceName) ;
			}
		}
		requestLocation = webserviceInfo.getRequestLocation() ;
		responseLocation = webserviceInfo.getResponseLocation() ;
		addressing = webserviceInfo.isAddressing() ;
	}

	public String getRequestName() {
		return requestName ;
	}
	
	public String getResponseName() {
		return responseName ;
	}
	
	public String getFaultName() {
		return faultName ;
	}
	
	public String getBindingName() {
		return bindingName ;
	}
	
	public String getServiceName() {
		return serviceName ;
	}
	
	public String getPortName() {
		return portName ;
	}
	
	public String getOperationName() {
		return operationName ;
	}
	
	public String getServletName() {
		return servletName ;
	}
	
	public String getServletPath() {
		return servletPath ;
	}
	
	public String getWSDLFileName() {
		return wsdlFileName ;
	}
	
	public String getPackageName() {
		return packageName ;
	}
	
	public String getClassName() {
		return className ;
	}
	
	public String getNamespace() {
		return namespace ;
	}
	
	public String getRequestAction() {
		return requestAction ;
	}
	
	public String getResponseAction() {
		return responseAction ;
	}
	
	public boolean isOneWay() {
		return isOneWay ;
	}
	
	public String getRequestLocation() {
		return requestLocation ;
	}
	
	public String getResponseLocation() {
		return responseLocation ;
	}
	
	public boolean isAddressing() {
		return addressing ;
	}
	
	static
	{
		DEFAULT_ENCODING = AccessController.doPrivileged(new PrivilegedAction<String>() {
			public String run()
			{
				return System.getProperty("file.encoding");
			}
		}) ;
	}
}
