/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.util;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.HashSet;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.xml.stream.XMLStreamException;

import org.apache.log4j.Logger;
import org.jboss.deployment.DeploymentException;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.publish.ContractReferencePublisher;
import org.jboss.internal.soa.esb.webservice.AddressingHandler;
import org.jboss.internal.soa.esb.webservice.ESBContractGenerator;
import org.jboss.internal.soa.esb.webservice.ESBResponseFilter;
import org.jboss.internal.soa.esb.webservice.ESBServiceEndpointInfo;
import org.jboss.internal.soa.esb.webservice.JAXWSProviderClassGenerator;
import org.jboss.mx.util.ObjectNameConverter;
import org.jboss.soa.esb.Service;
import org.jboss.soa.esb.listeners.config.WebserviceInfo;
import org.jboss.soa.esb.listeners.config.model.ModelAdapter;
import org.jboss.soa.esb.listeners.config.model.ModelParser;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * Util class for JBoss AS Deployers.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 * 
 */
public class JBossDeployerUtil
{
    private static final String PREFIX_CANONICAL_NAME = "jboss.esb:deployment=" ;
    private static final String ESB_ARTIFACT_NAME = "jbossesb.esb" ;
    
    private static final String HANDLER_CHAIN_PREFIX = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" + "<handler-chains xmlns=\"http://java.sun.com/xml/ns/javaee\" xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"" + " xsi:schemaLocation=\"http://java.sun.com/xml/ns/javaee javaee_web_services_1_2.xsd\">" + "<handler-chain><protocol-bindings>##SOAP11_HTTP</protocol-bindings>";
    private static final String HANDLER_CHAIN_POSTFIX = "</handler-chain></handler-chains>";
    private static final String HANDLER_PREFIX = "<handler><handler-name>";
    private static final String HANDLER_MIDDLE = "</handler-name><handler-class>";
    private static final String HANDLER_POSTFIX = "</handler-class></handler>";

    private static final String WSADDR_NAME = "WSAddressing Handler";
    private static final String WSADDR_HANDLER = "org.jboss.ws.extensions.addressing.jaxws.WSAddressingServerHandler";

    private static final String WSADDR_CONTEXT_NAME = "WSAddressing Context Handler";
    private static final String WSADDR_CONTEXT_HANDLER = AddressingHandler.class.getName();
    
    private static Logger log = Logger.getLogger(JBossDeployerUtil.class);

    private JBossDeployerUtil() { }

    public static String getHandlers(final ESBServiceEndpointInfo serviceInfo)
    {
        final String result;

        if (!serviceInfo.isAddressing())
        {
            result = null;
        } 
        else
        {
            final StringBuilder sb = new StringBuilder(HANDLER_CHAIN_PREFIX);
            if (serviceInfo.isAddressing())
            {
                sb.append(HANDLER_PREFIX).append(WSADDR_CONTEXT_NAME).append(HANDLER_MIDDLE).append(WSADDR_CONTEXT_HANDLER).append(HANDLER_POSTFIX);
                sb.append(HANDLER_PREFIX).append(WSADDR_NAME).append(HANDLER_MIDDLE).append(WSADDR_HANDLER).append(HANDLER_POSTFIX);
            }
            sb.append(HANDLER_CHAIN_POSTFIX);
            result = sb.toString();
        }
        return result;
    }

    public static File getESBWarFile(final String deploymentName, final String esbName, final boolean createDir, final File dir)
    {
        final File esbDir = new File(dir, deploymentName);
        if (!esbDir.exists())
        {
            if (!createDir)
            {
                return null;
            }
            esbDir.mkdir();
        }

        final int lastSeparator = esbName.lastIndexOf('.');
        final String warFileName = ((lastSeparator >= 0) ? esbName.substring(0, lastSeparator) : esbName) + ".war";

        return new File(esbDir, warFileName);
    }

    public static File createDir(final File parentDir, String dirName)
    {
        final File esbWarFiles = new File(parentDir, dirName);
        esbWarFiles.mkdirs();
        return esbWarFiles;
    }

    public static String readEsbConfig(final InputStream in) throws DeploymentException
    {
        try
        {
            final StringWriter sw = new StringWriter();
            XMLHelper.replaceSystemProperties(XMLHelper.getXMLStreamReader(in), XMLHelper.getXMLStreamWriter(sw));
            return sw.toString();
        } catch (XMLStreamException e)
        {
            throw new DeploymentException(e.getMessage(), e);
        } finally
        {
            try
            {
                in.close();
            } catch (IOException ignore)
            {
            }
        }
    }

    public static ModelAdapter getJbossEsbModel(final String configuration) throws DeploymentException
    {
        try
        {
            final StringReader sr = new StringReader(configuration);
            return ModelParser.getParser().parse(sr);
        } catch (final Exception ex)
        {
            throw new DeploymentException("Failed to parse jboss-esb.xml", ex);
        }
    }

    public static Properties getArtifactProperties(final String fileName) throws IOException
    {
        Properties properties = null;
        final InputStream in = ClassUtil.getResourceAsStream(fileName, JBossDeployerUtil.class);
        if (in != null)
        {
            log.debug("Loaded action artifact mapping from " + fileName);
            properties = new Properties();
            properties.load(in);
        }
        else
        {
            log.debug("No action artifact mapping") ;
        }
        return properties;
    }
    
    public static Set<ObjectName> getActionDependencies(final String deploymentName, final ModelAdapter model, final Properties actionArtifactProperties) throws DeploymentException
    {
        final Set<ObjectName> deps = new HashSet<ObjectName>();
        final Set<String> artifacts = new HashSet<String>() ;
        artifacts.add(ESB_ARTIFACT_NAME) ;
    
        final Set<String> actionClasses = model.getActions() ;
          
        final int numActionClasses = (actionClasses == null ? 0 : actionClasses.size()) ;
        if (numActionClasses > 0)
        {
            for(final String actionClass: actionClasses)
            {
                final String artifact = (String) actionArtifactProperties.get(actionClass) ;
                if (artifact != null)
                {
                    artifacts.add(artifact) ;
                }
            }
        }
          
        for(final String artifact: artifacts)
        {
            if (!deploymentName.equals(artifact))
            {
                final String canonicalName = PREFIX_CANONICAL_NAME + artifact ;
                try
                {
                    ObjectName on = ObjectNameConverter.convert(canonicalName);
                    deps.add(on) ;
                } 
                catch (MalformedObjectNameException e)
                {
                    throw new DeploymentException(e.getMessage(), e);
                }
            }
        }
        return deps;
    }

}
