/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.internal.soa.esb.util;

import org.apache.log4j.Logger;

import java.net.ServerSocket;
import java.net.Socket;
import java.io.*;

/**
 * Exit Listener utility class.
 * <p/>
 * Used to exit a process cleanly, when that process is forked as a "child" process
 * e.g. when forked by a &lt;java&gt; task in an Ant script.
 * <p/>
 * This is required because Windows (and other OSs - apparently Solaris is the same) doesn't
 * shutdown processes cleanly through their implementation of the {@link Process#destroy()}
 * method.  Apparently the Windows implementation uses the Windows shell "TerminateProcess"
 * call to kill all child processes and this does the job of termination rather brutally,
 * resulting in shutdown hooks not being triggered etc.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class Exit {

    private static Logger logger = Logger.getLogger(Exit.class);
    private static final String EXIT = "Exit";

    /**
     * Start the listener on the specified port.
     * @param port Listen port.
     */
    public static void startListener(final int port) throws Exception {
        new Thread() {
            public void run() {
                ServerSocket serverSocket = null;
                try {
                    serverSocket = new ServerSocket(port);
                    logger.info("Started Exit listener on port " + port);
                    while(true) {
                        Socket socket = serverSocket.accept();
                        OutputStream output = socket.getOutputStream();
                        InputStream input = socket.getInputStream();

                        try {
                            // Write an EXIT and see is it echoed back...
                            writeExit(output);
                            if(isExit(input)) {
                                // EXIT was echoed back, so we exit...
                                logger.info("Received process exit event on port " + port);
                                break;
                            }
                            // Not echoed back... must be someone else... continue to listen...
                        } finally {
                            closeSocket(output, input, socket);
                        }
                    }
                } catch (IOException e) {
                    System.err.println(e);
                    System.exit(1);
                    return;
                } finally {
                    if(serverSocket != null ) {
                        try {
                            serverSocket.close();
                        } catch (IOException e) {
                            System.err.println(e);
                        }
                    }
                }
                System.exit(0);
            }
        }.start();
    }

    public static void signalExit(int port) throws Throwable {
        Socket socket = new Socket("localhost", port);
        OutputStream output = socket.getOutputStream();
        InputStream input = socket.getInputStream();

        try {
            // Wait for an EXIT...
            if(isExit(input)) {
                // We got an EXIT... echo it back...
                writeExit(output);
            }
        } finally {
            closeSocket(output, input, socket);
        }
    }

    private static boolean isExit(InputStream input) {
        try {
            DataInputStream dataReader = new DataInputStream(input);
            return EXIT.equals(dataReader.readUTF());
        } catch(Exception e) {
            return false;
        }
    }


    private static void writeExit(OutputStream output) {
        try {
            DataOutputStream dataWriter = new DataOutputStream(output);

            dataWriter.writeUTF(EXIT);
            dataWriter.flush();
        } catch(Exception e) {            
        }
    }

    private static void closeSocket(OutputStream output, InputStream input, Socket socket) throws IOException {
        try {
            output.close();
        } finally {
            try {
                input.close();
            } finally {
                socket.close();
            }
        }
    }
}
