/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.internal.soa.esb.rosetta.pooling;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;

import javax.jms.JMSException;
import javax.jms.MessageConsumer;
import javax.jms.MessageProducer;
import javax.jms.QueueBrowser;
import javax.jms.TopicSubscriber;
import javax.jms.XASession;
import javax.transaction.Status;
import javax.transaction.Synchronization;
import javax.transaction.xa.XAResource;

import org.jboss.internal.soa.esb.rosetta.pooling.JmsConnectionPool.JmsSessionPool;
import org.jboss.soa.esb.common.TransactionStrategy;
import org.jboss.soa.esb.common.TransactionStrategyException;

/**
 * Wrapper for JMS XA session class, responsible for tracking resources and the pooling.
 */
class JmsXASession extends JmsSession implements Synchronization
{
    /**
     * The connection pool.
     */
    private final JmsConnectionPool connectionPool ;
    
    /**
     * The session delegate.
     */
    private final XASession session ;
    
    /**
     * Flag indicating whether this session is associated with a transaction.
     */
    private boolean associated ;
    
    /**
     * Cleanup actions
     */
    private enum Cleanup { close, release, none }
    
    /**
     * The cleanup action for the synchronization.
     */
    private Cleanup cleanupAction = Cleanup.none ;
    
    /**
     * Create the session wrapper.
     * @param connectionPool The current connection pool
     * @param sessionPool The current session pool
     * @param session The session delegate.
     * @param id The pool instance id.
     * @param requestedAcknowledgeMode The requested acknowledge mode for this session.
     * @throws JMSException
     */
    JmsXASession(final JmsConnectionPool connectionPool, final JmsSessionPool sessionPool, final XASession session, final long id, final int requestedAcknowledgeMode)
        throws JMSException
    {
        super(connectionPool, sessionPool, session, id, requestedAcknowledgeMode) ;
        this.connectionPool = connectionPool ;
        this.session = (XASession)getExceptionHandler(connectionPool, XASession.class, session) ;
    }
    
    @Override
    public void commit() throws JMSException
    {
        // Handled by the transaction
    }
    
    @Override
    public void rollback() throws JMSException
    {
        try
        {
            TransactionStrategy.getTransactionStrategy(true).rollbackOnly() ;
        }
        catch (final TransactionStrategyException tse)
        {
            final JMSException ex = new JMSException("Failed to rollback transaction") ;
            ex.initCause(tse) ;
            throw ex ;
        }
    }
    
    @Override
    protected MessageProducer getMessageProducer(MessageProducer messageProducer)
    {
        final InvocationHandler handler = new AssociationHandler(messageProducer) ;
        return (MessageProducer)Proxy.newProxyInstance(getClass().getClassLoader(), new Class[] {MessageProducer.class}, handler );
    }
    
    @Override
    protected MessageConsumer getMessageConsumer(MessageConsumer messageConsumer)
    {
        final InvocationHandler handler = new AssociationHandler(messageConsumer) ;
        return (MessageConsumer)Proxy.newProxyInstance(getClass().getClassLoader(), new Class[] {MessageConsumer.class}, handler);
    }
    
    @Override
    protected QueueBrowser getQueueBrowser(QueueBrowser queueBrowser)
    {
        final InvocationHandler handler = new AssociationHandler(queueBrowser) ;
        return (QueueBrowser)Proxy.newProxyInstance(getClass().getClassLoader(), new Class[] {QueueBrowser.class}, handler);
    }
    
    @Override
    protected TopicSubscriber getTopicSubscriber(TopicSubscriber topicSubscriber)
    {
        final InvocationHandler handler = new AssociationHandler(topicSubscriber) ;
        return (TopicSubscriber)Proxy.newProxyInstance(getClass().getClassLoader(), new Class[] {TopicSubscriber.class}, handler);
    }
    
    protected synchronized void handleCloseSession(final JmsConnectionPool jmsConnectionPool)
    {
        if (associated)
        {
            cleanupAction = Cleanup.close ;
        }
        else
        {
            connectionPool.handleCloseSession(this) ;
        }
    }
    
    protected synchronized void handleReleaseSession(JmsConnectionPool jmsConnectionPool)
    {
        if (associated)
        {
            cleanupAction = Cleanup.release ;
        }
        else
        {
            connectionPool.handleReleaseSession(this) ;
        }
    }
    
    protected synchronized void associate()
        throws JMSException
    {
        if (!associated)
        {
            cleanupAction = Cleanup.none ;
            final TransactionStrategy transactionStrategy = TransactionStrategy.getTransactionStrategy(true) ;
            try
            {
                transactionStrategy.registerSynchronization(this) ;
                setSuspect(true) ;
                final XAResource resource = session.getXAResource() ;
                transactionStrategy.enlistResource(resource) ;
            }
            catch (final TransactionStrategyException tse)
            {
                final JMSException ex = new JMSException("Failed to initialise transaction resources") ;
                ex.initCause(tse) ;
                throw ex ;
            }
            try
            {
                connectionPool.associateTransaction(this) ;
            }
            catch (final ConnectionException ce)
            {
                final JMSException ex = new JMSException("Failed to associate session with the current transaction") ;
                ex.initCause(ce) ;
                throw ex ;
            }
            
            setSuspect(false) ;
            associated = true ;
        }
    }
    
    public void beforeCompletion()
    {
    }
    
    public synchronized void afterCompletion(final int result)
    {
        connectionPool.disassociateTransaction(this) ;
        switch (cleanupAction)
        {
        case close:
            if (result == Status.STATUS_COMMITTED)
            {
                connectionPool.handleCloseSession(this) ;
                break ;
            }
            // fall through
        case release:
            connectionPool.handleReleaseSession(this) ;
            break ;
        case none:
            // Reference held by caller
        }
        associated = false ;
    }
    
    /**
     * Handler responsible for associating XA resources.
     * @author kevin
     */
    private final class AssociationHandler implements InvocationHandler
    {
        /**
         * The target instance.
         */
        private final Object target ;
        
        /**
         * Construct the handler using the specified target.
         * @param target The target instance.
         */
        public AssociationHandler(final Object target)
        {
            this.target = target ;
        }
        
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable
        {
            associate() ;
            try
            {
                return method.invoke(target, args);
            }
            catch (final InvocationTargetException ite)
            {
                throw ite.getCause() ;
            }
        }
    }
}
