/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2009
 */
package org.jboss.internal.soa.esb.publish;

import java.util.HashMap;
import java.util.Map;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.lifecycle.LifecyclePriorities;
import org.jboss.soa.esb.lifecycle.LifecycleResource;
import org.jboss.soa.esb.lifecycle.LifecycleResourceException;
import org.jboss.soa.esb.lifecycle.LifecycleResourceFactory;
import org.jboss.soa.esb.lifecycle.LifecycleResourceManager;
import org.jboss.soa.esb.listeners.ListenerTagNames;

/**
 * ContractProviderLifecycleResource.
 * 
 * @author dward at jboss.org
 */
public class ContractProviderLifecycleResource
{
	
	private static final Logger logger = Logger.getLogger(ContractProviderLifecycleResource.class);
	
	private static final LifecycleResource<Map<String, ContractProvider>> lifecycleContractProviders =
		new LifecycleResource<Map<String, ContractProvider>>(new LifecycleContractProviderFactory(), LifecyclePriorities.CONTRACT_PROVIDER_PRIORITY);
	
	private ContractProviderLifecycleResource() {}
	
	public static void putContractProvider(String serviceCategory, String serviceName, ContractProvider contractProvider)
		throws LifecycleResourceException
	{
        AssertArgument.isNotNull(contractProvider, "contractProvider");
		String key = buildKey(serviceCategory, serviceName);
		Map<String, ContractProvider> map = lifecycleContractProviders.getLifecycleResource();
		synchronized(map)
		{
			map.put(key, contractProvider);
		}
	}
	
	public static ContractProvider getContractProvider(String serviceCategory, String serviceName)
		throws LifecycleResourceException
	{
		String key = buildKey(serviceCategory, serviceName);
		Map<String, ContractProvider> map = lifecycleContractProviders.getLifecycleResource();
		synchronized(map)
		{
			return map.get(key);
		}
	}
	
	private static String buildKey(String serviceCategory, String serviceName)
	{
        AssertArgument.isNotNullAndNotEmpty(serviceCategory, "serviceCategory");
        AssertArgument.isNotNullAndNotEmpty(serviceName, "serviceName");
		StringBuilder sb = new StringBuilder();
		sb.append(ContractProvider.class.getSimpleName());
		sb.append("?deployment=");
		sb.append(LifecycleResourceManager.getSingleton().getIdentity());
		sb.append('&');
		sb.append(ListenerTagNames.SERVICE_CATEGORY_NAME_TAG);
		sb.append('=');
		sb.append(serviceCategory);
		sb.append('&');
		sb.append(ListenerTagNames.SERVICE_NAME_TAG);
		sb.append('=');
		sb.append(serviceName);
		return sb.toString();
	}
	
	private static class LifecycleContractProviderFactory implements LifecycleResourceFactory<Map<String, ContractProvider>>
	{
		
		public Map<String, ContractProvider> createLifecycleResource(String lifecycleIdentity) throws LifecycleResourceException
		{
			return new HashMap<String, ContractProvider>();
		}
		
		public void destroyLifecycleResource(Map<String, ContractProvider> resource, String lifecycleIdentity) throws LifecycleResourceException
		{
			if (resource.size() > 0)
			{
				if ( logger.isInfoEnabled() )
				{
					for ( String key : resource.keySet() )
					{
						logger.info("Removing " + key);
					}
				}
				resource.clear();
			}
		}
        
	}

}
