package org.jboss.internal.soa.esb.message.format.xml.marshal;

import java.io.IOException;
import java.io.Serializable;
import java.net.URI;
import java.net.URISyntaxException;

import org.jboss.internal.soa.esb.util.Encoding;
import org.jboss.soa.esb.MarshalException;
import org.jboss.soa.esb.UnmarshalException;

/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */

/**
 * Used to plug in new Object marshal/unmarshal formats dynamically. When
 * packing objects in XML, the system runs through the list of registered
 * plug-ins until it finds one that can deal with the object type (or faults).
 * When packing, the name (type) of the plug-in that packed the object is also
 * attached to facilitate unpacking.
 * 
 * @author Mark Little
 * 
 */

public class SerializedMarshalUnmarshalPlugin implements MarshalUnmarshalPlugin
{
	public SerializedMarshalUnmarshalPlugin()
	{
		try
		{
			_type = new URI("urn:xml/marshalunmarshal/plugin/serialization");
		}
		catch (URISyntaxException ex)
		{
			ex.printStackTrace();
		}
	}

	/**
	 * Can the plugin pack the specified object?
	 * @param value The object to pack.
	 * @return true if the object can be packed, false otherwise.
	 */
        public boolean canPack(final Object value)
        {
            return (value instanceof Serializable) ;
        }
        
        /**
         * Pack the provided object into the document.
         * 
         * @param param the object to pack.
         * @return the packed version of the object.
         * 
         * @throws MarshalException thrown if there is a problem packing.
         */
        public String marshal (Object param)
            throws MarshalException
	{
	    if (param instanceof Serializable)
            {
	        try
	        {
	            return Encoding.encodeObject((Serializable)param) ;
	        }
	        catch (final IOException ioe)
	        {
                    throw new MarshalException("Failed to encode value", ioe) ;
	        }
            }
            else
            {
                throw new MarshalException("Invalid value type for marshaling plugin:" + param.getClass().getName()) ;
            }
	}

        /**
         * Unpack the object from the document.
         * 
         * @param content the object content.
         * 
         * @return the object, or <code>null</code> if this implementation cannot deal with the
         * format.
         * @throws UnmarshalException thrown if there is a problem unpacking.
         */
        public Object unmarshal (final String content)
            throws UnmarshalException
	{
	    try
	    {
	        return Encoding.decodeToObject(content) ;
	    }
            catch (final ClassNotFoundException cnfe)
            {
                throw new UnmarshalException("Failed to decode value", cnfe) ;
            }
            catch (final IOException ioe)
            {
                throw new UnmarshalException("Failed to decode value", ioe) ;
            }
	}

	/**
	 * @return the unique name for this plugin.
	 */

	public URI type()
	{
		return _type;
	}

	private URI _type = null;
}
