/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 */

package org.jboss.internal.soa.esb.message.format.xml;

import java.io.IOException;
import java.io.Serializable;

import org.jboss.internal.soa.esb.message.format.DeferredDeserialisationException;
import org.jboss.internal.soa.esb.util.Encoding;

/**
 * Wrapper class for serialised values, supporting JIT deserialisation.
 * 
 * @author <a href='kevin.conner@jboss.com'>Kevin Conner</a>
 */

public class SerializedValueImpl
{
    /**
     * The value.
     */
    private Serializable value ;
    
    /**
     * The serialised form.
     */
    private String serialisedForm ;
    
    /**
     * Construct the serialised value wrapper for the specific value.
     * @param value The serializable value.
     */
    public SerializedValueImpl(final Serializable value)
    {
        this.value = value ;
    }
    
    /**
     * Construct the serialised value wrapper for the specific serialised form.
     * @param serialisedForm The serialised form.
     */
    public SerializedValueImpl(final String serialisedForm)
    {
        this.serialisedForm = serialisedForm ;
    }
    
    /**
     * Get the wrapped value.
     * @return The wrapped value.
     */
    public Serializable getValue()
    {
        if ((value == null) && (serialisedForm != null))
        {
            try
            {
                value = Encoding.decodeToObject(serialisedForm) ;
            }
            catch (final IOException ioe)
            {
                throw new DeferredDeserialisationException("Error reading object input stream", ioe) ;
            }
            catch (final ClassNotFoundException cnfe)
            {
                throw new DeferredDeserialisationException("Error constructing object value", cnfe) ;
            }
        }
        serialisedForm = null ;
        return value ;
    }
    
    /**
     * Get the serialised form.
     * @return The serialised form.
     */
    public String getSerialisedForm()
    {
        if ((serialisedForm == null) && (value != null))
        {
            try
            {
                serialisedForm = Encoding.encodeObject(value) ;
            }
            catch (final IOException ioe)
            {
                throw new DeferredDeserialisationException("Error writing object input stream", ioe) ;
            }
        }
        return serialisedForm ;
    }
    
    /**
     * Return a string representation of this object.
     * @return the string representation of the value or a deferred identifier.
     */
    public String toString()
    {
        if (value != null)
        {
            return value.toString() ;
        }
        else
        {
            return "Deferred serialized value: " + Integer.toHexString(System.identityHashCode(this)) ;
        }
    }
}
