/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author schifest@heuristica.com.ar
 */
package org.jboss.internal.soa.esb.message.format.xml;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map.Entry;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.util.Encoding;
import org.jboss.internal.soa.esb.util.stax.ElementContent;
import org.jboss.internal.soa.esb.util.stax.StreamHelper;
import org.jboss.internal.soa.esb.util.stax.TextElement;
import org.jboss.soa.esb.message.Properties;

public class PropertiesImpl extends ElementContent implements Properties
{
    /**
     * The default constructor
     */
    public PropertiesImpl()
    {
    }
    
    /**
     * Construct a set of properties from the input stream.
     * 
     * @param in The input stream.
     * @throws XMLStreamException For errors during parsing.
     */
    public PropertiesImpl(final XMLStreamReader in)
        throws XMLStreamException
    {
        parse(in) ;
    }

	public synchronized Object getProperty(String name)
	{
		return unwrap(_table.get(name));
	}

	public Object getProperty(String name, Object defaultVal)
	{
		Object oRet = getProperty(name);
		return (null == oRet) ? defaultVal : oRet;
	}

	public synchronized Object setProperty(String name, Object value)
	{
        AssertArgument.isNotNull(name, "name");
        AssertArgument.isNotNull(value, "value");

        if (!(value instanceof Serializable)) {
			throw new IllegalArgumentException("Value of property '" + name + "' must implement " + Serializable.class.getName() + ".  Value is of type " + value.getClass().getName());
        }

        return unwrap(_table.put(name, new SerializedValueImpl((Serializable)value)));
    }

	public synchronized Object remove(String name)
	{
		return unwrap(_table.remove(name));
	}

	public synchronized int size()
	{
		return _table.size();
	}

	public synchronized String[] getNames()
	{
		return _table.keySet().toArray(new String[0]);
	}

        /**
         * Write the child content of the element.
         * @param out The output stream.
         * @throws XMLStreamException For errors during output.
         */
	@Override
	protected synchronized void writeChildContent(XMLStreamWriter out)
	        throws XMLStreamException
	{
	    for (Entry<String, SerializedValueImpl> entry: _table.entrySet())
	    {
	        final String origPropertyURI = StreamHelper.writeStartElement(out, XMLUtil.ESB_QNAME_PROPERTY) ;
	        
	        final TextElement keyElement = new TextElement(Encoding.encodeBytes(entry.getKey().getBytes())) ;
	        StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_PROPERTY_KEY, keyElement) ;
	        
	        final String value = entry.getValue().getSerialisedForm() ;
	        final TextElement valueElement = new TextElement(value) ;
	        StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_PROPERTY_VALUE, valueElement) ;
	        
	        StreamHelper.writeEndElement(out, XMLUtil.ESB_QNAME_PROPERTY.getPrefix(), origPropertyURI) ;
	    }
	}
	
        /**
         * Add the element.
         * @param in The current input stream.
         * @param elementName The qualified element name.
         * @throws XMLStreamException For errors during parsing.
         */
        @Override
        protected void putElement(final XMLStreamReader in,
            final QName elementName)
            throws XMLStreamException
        {
            if (XMLUtil.ESB_QNAME_PROPERTY.equals(elementName))
            {
                StreamHelper.checkNextStartTag(in, XMLUtil.ESB_QNAME_PROPERTY_KEY) ;
                final TextElement keyElement = new TextElement(in) ;
                final String key = new String(Encoding.decodeToBytes(keyElement.getText())) ;
                
                StreamHelper.checkNextStartTag(in, XMLUtil.ESB_QNAME_PROPERTY_VALUE) ;
                final TextElement valueElement = new TextElement(in) ;
                final SerializedValueImpl value = new SerializedValueImpl(valueElement.getText()) ;
                
                StreamHelper.checkParentFinished(in) ;
                
                _table.put(key, value) ;
            }
            else
            {
                throw new XMLStreamException("Unexpected element name: " + elementName) ;
            }
        }
        
	public synchronized String toString()
	{
		return "properties: [ "+_table.toString()+" ]";
	}

	private Object unwrap(final SerializedValueImpl value)
	{
	    return (value == null ? null : value.getValue()) ;
	}
	
	private HashMap<String, SerializedValueImpl> _table = new HashMap<String, SerializedValueImpl>();

}
