/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated 
 * by the @authors tag. All rights reserved. 
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors. 
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, 
 * MA  02110-1301, USA.
 * 
 * (C) 2005-2006,
 * @author mark.little@jboss.com
 */
package org.jboss.internal.soa.esb.message.format.xml;

import java.io.IOException;
import java.io.Serializable;
import java.net.URI;

import javax.xml.namespace.QName;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.stream.XMLStreamWriter;

import org.jboss.internal.soa.esb.util.stax.ElementContent;
import org.jboss.internal.soa.esb.util.stax.StreamHelper;
import org.jboss.soa.esb.addressing.Call;
import org.jboss.soa.esb.message.Attachment;
import org.jboss.soa.esb.message.Body;
import org.jboss.soa.esb.message.ByReferenceMessage;
import org.jboss.soa.esb.message.Context;
import org.jboss.soa.esb.message.Fault;
import org.jboss.soa.esb.message.Header;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.Properties;
import org.jboss.soa.esb.message.body.content.Payload;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.message.format.MessageType;
import org.jboss.soa.esb.util.Util;

/**
 * This is the basic internal core message abstraction. A message consists of
 * the following components:
 * 
 * Header: the header information contains information such as the destination
 * EPR, the sender EPR, where the reply goes etc, i.e., general message-level
 * functional information. Context: additional information to contextualise the
 * message; for example, transaction or security data, the identity of the
 * ultimate receiver, or HTTP-cookie like information. Body: the actual payload
 * of the message. Fault: any fault information associated with the message.
 * Attachment: any attachments associated with the message. Properties: any
 * properties associated with the message.
 * 
 * Each message, once created, has a corresponding element for these 5
 * components. That element may be empty (<b>NOT NULL</b>). The object
 * representing the element can then be used to act on the corresponding data
 * item in the message.
 * 
 * @author Mark Little
 * 
 */

public class MessageImpl extends ElementContent implements ByReferenceMessage
{
	public MessageImpl()
	{
		_theHeader = new HeaderImpl();
		_theContext = new ContextImpl();
		_theBody = new BodyImpl();
		_theFault = new FaultImpl(_theBody);
		_theAttachment = new AttachmentImpl();
		_theProperties = new PropertiesImpl();
	}
	
	private MessageImpl(final MessageImpl message)
	{
		// Header and context are not by reference.
		_theHeader = new HeaderImpl() ;
		final Call origCall = message._theHeader.getCall() ;
		final Call call = new Call(origCall.getTo()) ;
		call.setAction(origCall.getAction()) ;
		call.setFaultTo(origCall.getFaultTo()) ;
		call.setFrom(origCall.getFrom()) ;
		call.setMessageID(origCall.getMessageID()) ;
		call.setRelatesTo(origCall.getRelatesTo()) ;
		call.setReplyTo(origCall.getReplyTo()) ;
		_theHeader.setCall(call) ;
		_theContext = new ContextImpl(message._theContext) ;
		
		_theBody = message._theBody ;
		_theFault = message._theFault ;
		_theAttachment = message._theAttachment ;
		_theProperties = message._theProperties ;
	}
        /**
         * Construct a message from the input stream.
         * 
         * @param in The input stream.
         * @throws XMLStreamException For errors during parsing.
         */
	public MessageImpl(final XMLStreamReader in)
	    throws XMLStreamException
        {
	    this() ;
	    parse(in) ;
	    
	    // Replace the body
            final String payloadType = Payload.bodyType(this);

            if ((payloadType != null)
                            && (!payloadType.equals(Payload.RAW_BODY)))
            {
                _theBody = (BodyImpl) MessageFactory.getInstance()
                                .createBodyType(this, payloadType);
            }
            _theFault = new FaultImpl(_theBody) ;
        }

	/**
	 * @return get the header component of the message.
	 */

	public Header getHeader()
	{
		return _theHeader;
	}

	/**
	 * @return get the context component of the message.
	 */

	public Context getContext()
	{
		return _theContext;
	}

	/**
	 * @return get the body component of the message.
	 */

	public Body getBody()
	{
		return _theBody;
	}

	/**
	 * @return get any faults associated with the message. These should not be
	 *         application level faults, but comms level.
	 */

	public Fault getFault()
	{
		return _theFault;
	}

	/**
	 * @return get any message attachments.
	 */

	public Attachment getAttachment()
	{
		return _theAttachment;
	}

	/**
	 * @return the type of this message format.
	 */

	public URI getType()
	{
		return MessageType.JBOSS_XML;
	}

	/**
	 * getProperties()
	 * 
	 * @return org.jboss.soa.esb.message.Properties - any message properties.
	 */
	public Properties getProperties()
	{
		return _theProperties;
	}

	/**
	 * Return a duplicate of this instance. Just serialize and then
	 * deserialize.
	 */

    public Message copy () throws IOException
    {
        try
        {
            Serializable state = Util.serialize(this);
            
            return (MessageImpl) Util.deserialize(state);
        }
        catch (final IOException ioe)
        {
            throw ioe ;
        }
        catch (Exception ex)
        {
            final IOException ioe = new IOException("Unexpected exception during copying") ;
            ioe.initCause(ex) ;
            throw ioe ;
        }
    }

	public String toString()
	{
		return "message: [ JBOSS_XML ]\n" + _theHeader.toString() + "\n"
				+ _theContext.toString() + "\n" + _theBody.toString() + "\n"
				+ _theFault.toString() + "\n" + _theAttachment.toString()
				+ "\n" + _theProperties.toString();
	}

	// to/from XML here, rather than on individual elements

        /**
         * Write the child content of the element.
         * @param out The output stream.
         * @throws XMLStreamException For errors during output.
         */
	public void writeChildContent(final XMLStreamWriter out)
	    throws XMLStreamException
	{
	    StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_HEADER, _theHeader) ;
            StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_CONTEXT, _theContext) ;
            StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_BODY, _theBody) ;
            StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_ATTACHMENT, _theAttachment) ;
            StreamHelper.writeElement(out, XMLUtil.ESB_QNAME_PROPERTIES, _theProperties) ;
	}
	
        /**
         * Add the element.
         * @param in The current input stream.
         * @param elementName The qualified element name.
         */
        protected void putElement(final XMLStreamReader in, final QName elementName)
            throws XMLStreamException
        {
            if (XMLUtil.ESB_QNAME_HEADER.equals(elementName))
            {
                _theHeader = new HeaderImpl(in) ;
            }
            else if (XMLUtil.ESB_QNAME_CONTEXT.equals(elementName))
            {
                _theContext = new ContextImpl(in) ;
            }
            else if (XMLUtil.ESB_QNAME_BODY.equals(elementName))
            {
                _theBody = new BodyImpl(in) ;
            }
            else if (XMLUtil.ESB_QNAME_ATTACHMENT.equals(elementName))
            {
                _theAttachment = new AttachmentImpl(in) ;
            }
            else if (XMLUtil.ESB_QNAME_PROPERTIES.equals(elementName))
            {
                _theProperties = new PropertiesImpl(in) ;
            }
            else
            {
                throw new XMLStreamException("Unexpected element name: " + elementName) ;
            }
        }

        /**
         * Create the message used for pass by reference semantics.
         * @return the referencable message.
         */
        public Message reference()
        {
            return new MessageImpl(this) ;
        }

	// should be a capability on the base interface, but no changes for 4.2
	// ...

	void replaceBody(BodyImpl body)
	{
		if (body == null)
			throw new IllegalArgumentException();

		_theBody = body;
	}

	private HeaderImpl _theHeader;

	private ContextImpl _theContext;

	private BodyImpl _theBody;

	private FaultImpl _theFault;

	private AttachmentImpl _theAttachment;

	private PropertiesImpl _theProperties;
}
