/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.internal.soa.esb.message.filter;

import java.io.File;
import java.util.Map;

import org.jboss.soa.esb.addressing.eprs.JDBCEpr;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.couriers.CourierException;
import org.jboss.soa.esb.filter.FilterManager;
import org.jboss.soa.esb.filter.InputOutputFilter;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerUtil;
import org.jboss.soa.esb.message.Message;

/**
 * An inputoutput filter is called when the message is being received/sent and
 * after it has been received from the transport/before it is passed to the
 * transport. Implementations may augment or modify the message en route.
 * Implementations that are pure output or pure input can just override the
 * desired operation.
 * 
 * This one adds some metadata to the message as it flows through the ESB.
 * 
 * @author marklittle
 */

public class GatewayFilter extends InputOutputFilter
{
    public Message onOutput (Message msg, Map<String, Object> params)
	    throws CourierException
    {
	if (params != null)
	{
	    /*
                 * We don't need to check to see if we're deployed on a gateway.
                 * If we aren't, then none of these objects will be present
                 * anyway!
                 */

	    File inFile = (File) params.get(Environment.ORIGINAL_FILE);

	    if (inFile != null)
		msg.getProperties().setProperty(
			Environment.ORIGINAL_FILE_NAME_MSG_PROP,
			inFile.getName());

	    ConfigTree config = (ConfigTree) params
		    .get(Environment.GATEWAY_CONFIG);

	    if (config != null)
	    {
		try
		{
        		String url = ListenerUtil.getValue(config, JDBCEpr.URL_TAG, null);
        
        		if (url != null)
        		    msg.getProperties().setProperty(
        			    Environment.ORIGINAL_URL_PROP, url);
		}
		catch (Throwable ex)
		{
		    FilterManager._logger.warn("GatewayFilter encountered problem during JDBC config check.", ex);
		}

		try
		{
        		String queueName = ListenerUtil.getValue(config,
        			JMSEpr.DESTINATION_NAME_TAG, null);
        
        		if (queueName != null)
        		    msg.getProperties()
        			    .setProperty(
        				    Environment.ORIGINAL_QUEUE_NAME_MSG_PROP,
        				    queueName);
		}
		catch (Throwable ex)
		{
		    FilterManager._logger.warn("GatewayFilter encountered problem during JMS config check.", ex);
		}
	    }
	}

	return msg;
    }
}
