/**
 * 
 */
package org.jboss.soa.esb.oracle.aq;


import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;

import java.io.FileInputStream;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;

import javax.jms.MessageConsumer;
import javax.jms.MessageProducer;
import javax.jms.Queue;
import javax.jms.QueueConnection;
import javax.jms.QueueConnectionFactory;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.jms.TextMessage;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.sql.DataSource;

import oracle.jms.AQjmsFactory;
import oracle.jms.AQjmsSession;

import org.junit.Before;
import org.junit.Test;
import org.mockejb.jndi.MockContextFactory;

/**
 * @author kstam
 *
 */
public class QueueTest 
{
    private final static String Q_NAME = "org.jboss.soa.esb.oracle.aq.q.name";
    private final static String MESSAGE_CONTENT="Oracle AQ message content";
    private String user, password, server, instance, schema, driver, portStr, qName;
    private int port;
    /**
     * Tests for completeness of the properties needed ro make a db connection.
     *
     */
    @Test
    public void checkPropertiesInFile() 
    {
            System.out.println("user=" + user + ", password=" + password + ", server=" + server
                    + ", port=" + portStr + ", qName=" + qName
                    + ", instance=" + instance + " schema=" + schema + ", driver=" + driver);
            assertNotNull(user);
            assertNotNull(password);
            assertNotNull(server);
            assertNotNull(portStr);
            assertFalse(port==0);
            assertNotNull(instance);
            assertNotNull(schema);
            assertNotNull(driver);   
            assertNotNull(qName);
    }
    /**
     * Tries to make a queueConnection to Oracle using the properties in the
     * oracle-aq.properties file.
     *
     */
    @Test
    public void createQueueConnection()
    {
        QueueConnection queueConnection = null;
        Session session = null;
        System.out.println("Building our own db connection");
        try {
            DataSource ds = AQUtil.getSQLDataSource(server, instance, port, driver, user, password);
            QueueConnectionFactory queueFactory = AQjmsFactory.getQueueConnectionFactory(ds);
            queueConnection = queueFactory.createQueueConnection();
            assertNotNull(queueConnection);
            session = (AQjmsSession) queueConnection.createQueueSession(false,
                    Session.AUTO_ACKNOWLEDGE);
            assertNotNull(session);
        } catch (Throwable e) {
            e.printStackTrace();
            assertTrue(false);
        } finally {
            if (session!=null) {
                try { session.close();
                } catch (Exception e) {}
            }
            if (queueConnection!=null) {
                try { queueConnection.close();
                } catch (Exception e) {}
            }
        }
    }
    /**
     * Tries to locate the specified Q in Oracle using the properties in the
     * oracle-aq.properties file.
     *
     */
    @Test
    public void findGivenQueue()
    {
        boolean foundQueue=false;
        QueueConnection queueConnection = null;
        Session session = null;
        System.out.println("Building our own db connection");
        try {
            DataSource ds = AQUtil.getSQLDataSource(server, instance, port, driver, user, password);
            QueueConnectionFactory queueFactory = AQjmsFactory.getQueueConnectionFactory(ds);
            queueConnection = queueFactory.createQueueConnection();
            assertNotNull(queueConnection);
            session = (AQjmsSession) queueConnection.createQueueSession(false,
                    Session.AUTO_ACKNOWLEDGE);
            assertNotNull(session);
            
            List<DestinationInfo> list = AQUtil.getDestinationInfoList(ds, schema);
            // for each queue
            Iterator<DestinationInfo> iter = list.iterator();
            while (iter.hasNext()) {
                DestinationInfo di = iter.next();
                System.out.println("Found Destination: " + di.getQueueName());
                // register normal queue
                if (di.getQueueName().equalsIgnoreCase(qName)) {
                    System.out.println("Matched up given Queue: " + qName + " with " + di.getQueueName());
                    foundQueue=true;
                }
            }
            assertTrue(foundQueue);
            
        } catch (Throwable e) {
            e.printStackTrace();
            assertTrue(false);
        } finally {
            if (session!=null) {
                try { session.close();
                } catch (Exception e) {}
            }
            if (queueConnection!=null) {
                try { queueConnection.close();
                } catch (Exception e) {}
            }
        }
    }
    /**
     * Sends a message to the specified Q in Oracle and receives the same message.
     */
    public void drainQueue()
    {
        QueueConnection connection = null;
        QueueSession session = null;
        MessageConsumer messageConsumer = null;
        try {
            Hashtable<String,String> env = new Hashtable<String, String>();
            env.put(AQInitialContextFactory.DB_SERVER, server);
            env.put(AQInitialContextFactory.DB_INSTANCE, instance);
            env.put(AQInitialContextFactory.DB_PORT, portStr);
            env.put(AQInitialContextFactory.DB_USER, user);
            env.put(AQInitialContextFactory.DB_PASSWD, password);
            env.put(AQInitialContextFactory.DB_SCHEMA, schema);
            env.put(AQInitialContextFactory.DB_DRIVER, driver);
            
            MockContextFactory.setAsInitial();
            AQInitialContextFactory factory = new AQInitialContextFactory();
            Context context = factory.getInitialContext(env);
            QueueConnectionFactory queueConnectionFactory = (QueueConnectionFactory) 
                context.lookup(AQInitialContextFactory.QUEUE_CONNECTION_FACTORY);
            connection = queueConnectionFactory.createQueueConnection();
            session = connection.createQueueSession(false,
            QueueSession.AUTO_ACKNOWLEDGE);
            Queue queue = (Queue) context.lookup(qName);
            messageConsumer = session.createReceiver(queue);
            connection.start();
            TextMessage receiveMessage = session.createTextMessage(MESSAGE_CONTENT);
            while (receiveMessage!=null) {
                receiveMessage = (TextMessage) messageConsumer.receive(10);
                if (receiveMessage!=null) System.out.println("Drain:" + receiveMessage.getText());
            }   
        } catch (Exception e) {
            e.printStackTrace();
            assertTrue(false);
        } finally {
            if (messageConsumer!=null) {
                try {  messageConsumer.close();
                } catch (Exception e) {}
            }
            if (session!=null) {
                try { session.close();
                } catch (Exception e) {}
            }
            if (connection!=null) {
                try { connection.close();
                } catch (Exception e) {}
            }
        }
    }
    /**
     * Sends a message to the specified Q in Oracle and receives the same message.
     */
    @Test
    public void sendAndReceiveMessage()
    {
        drainQueue();
        QueueConnection connection = null;
        QueueSession session = null;
        MessageProducer messageProducer = null;
        MessageConsumer messageConsumer = null;
        try {
            Properties properties = new Properties();
            properties.put(Context.INITIAL_CONTEXT_FACTORY, "org.jboss.soa.esb.oracle.aq.AQInitialContextFactory");
            properties.put(AQInitialContextFactory.DB_SERVER, server);
            properties.put(AQInitialContextFactory.DB_INSTANCE, instance);
            properties.put(AQInitialContextFactory.DB_PORT, portStr);
            properties.put(AQInitialContextFactory.DB_USER, user);
            properties.put(AQInitialContextFactory.DB_PASSWD, password);
            properties.put(AQInitialContextFactory.DB_SCHEMA, schema);
            properties.put(AQInitialContextFactory.DB_DRIVER, driver);
            MockContextFactory.setAsInitial();
            Context context = new InitialContext(properties);
            QueueConnectionFactory queueConnectionFactory = (QueueConnectionFactory) 
                context.lookup(AQInitialContextFactory.QUEUE_CONNECTION_FACTORY);
            connection = queueConnectionFactory.createQueueConnection();
            session = connection.createQueueSession(false,
            QueueSession.AUTO_ACKNOWLEDGE);
            Queue queue = (Queue) context.lookup(qName);
            messageProducer = session.createSender(queue);
            messageConsumer = session.createReceiver(queue);
            connection.start();
            TextMessage sendMessage = session.createTextMessage(MESSAGE_CONTENT);
            messageProducer.send(sendMessage);
            TextMessage receiveMessage = (TextMessage) messageConsumer.receive(2000);
            String receiveContent = receiveMessage.getText();
            assertEquals(MESSAGE_CONTENT, receiveContent);
            
        } catch (Exception e) {
            e.printStackTrace();
            assertTrue(false);
        } finally {
            if (messageProducer!=null) {
                try {  messageProducer.close();
                } catch (Exception e) {}
            }
            if (messageConsumer!=null) {
                try {  messageConsumer.close();
                } catch (Exception e) {}
            }
            if (session!=null) {
                try { session.close();
                } catch (Exception e) {}
            }
            if (connection!=null) {
                try { connection.close();
                } catch (Exception e) {}
            }
        }
    }
    
    @Before
    public void readProperyFile() {
        try {
            Properties properties = new Properties();
            properties.load(new FileInputStream("src/test/resources/oracle-aq.properties"));
            if (properties.containsKey(AQInitialContextFactory.DB_DATASOURCE)) {
                String datasourceName = properties.getProperty(AQInitialContextFactory.DB_DATASOURCE);
                System.out.println("Found datasource name: " + datasourceName);
                System.err.println("We can't use a Datasource in JUnit.");
                assertTrue(false);
            } else {
                user     = properties.getProperty(AQInitialContextFactory.DB_USER);
                password = properties.getProperty(AQInitialContextFactory.DB_PASSWD);
                server   = properties.getProperty(AQInitialContextFactory.DB_SERVER);
                instance = properties.getProperty(AQInitialContextFactory.DB_INSTANCE);
                schema   = properties.getProperty(AQInitialContextFactory.DB_SCHEMA);
                driver   = properties.getProperty(AQInitialContextFactory.DB_DRIVER);
                portStr  = properties.getProperty(AQInitialContextFactory.DB_PORT);
                port     = Integer.valueOf(portStr);
                qName    = properties.getProperty(Q_NAME);
            }
        } catch (Exception e) {
            e.printStackTrace();
            assertTrue(false);
        }
    }
    
    
}
