/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.xml;

import org.xml.sax.SAXException;
import org.w3c.dom.Document;
import org.jboss.logging.Logger;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.validation.Validator;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Schema;
import javax.xml.transform.Source;
import javax.xml.transform.stream.StreamSource;
import javax.xml.transform.dom.DOMSource;
import javax.xml.XMLConstants;
import java.io.File;

/**
 * A Helper classes used to validate xml files against supplied schemas.
 * 
 * @author $Revision$ $Id$
 */
public class XmlValidation
{

	/**
	 * Our Logger
	 */
	private Logger log = Logger.getLogger(XmlValidation.class);

	/**
	 * The xml base directory.
	 */
	private String xmlBaseDirectory = null;

	/**
	 * The xml base directory key.
	 */
	private String xmlBaseDirectoryKey = "xml.base.dir";

	/**
	 * The xsd base directory.
	 */
	private String xsdBaseDirectory = null;

	/**
	 * The xsd base directory key.
	 */
	private String xsdBaseDirectoryKey = "xsd.base.dir";

	/**
	 * The path separator
	 */
	private static String SEP = System.getProperty("file.separator");

	/**
	 * Default class constructor
	 */
	public XmlValidation ()
	{
		if (System.getProperty(xmlBaseDirectoryKey) != null)
		{
			xmlBaseDirectory = System.getProperty(xmlBaseDirectoryKey);
			log.info("Using xml base directory " + xmlBaseDirectory);
		}
		if (System.getProperty(xsdBaseDirectoryKey) != null)
		{
			xsdBaseDirectory = System.getProperty(xsdBaseDirectoryKey);
			log.info("Using xsd base directory " + xsdBaseDirectory);
		}
	}

	/**
	 * Validation method used to validate an xml file against an xsd.
	 * 
	 * @param xml
	 *            The xml file to be validated.
	 * @param xsd
	 *            The schema to validate against.
	 * @return Boolean true/false indicating successful validation.
	 * @throws SAXException
	 *             Failure during validation.
	 */
	public boolean validate (String xml, String xsd) throws SAXException
	{
		if (log.isInfoEnabled())
		{
			log.info("Validating " + xml + " Against " + xsd);
		}
		boolean isValid = false;
		File xmlFile = new File(xmlBaseDirectory + SEP + xml);
		File xsdFile = new File(xsdBaseDirectory + SEP + xsd);
		log.info("Xml File=" + xmlFile.getAbsolutePath());
		log.info("Xsd File=" + xsdFile.getAbsolutePath());
		DocumentBuilder parser = DocumentBuilderFactory.newInstance()
				.newDocumentBuilder();
		Document document = parser.parse(xmlFile);
		// log.info("Xml=" + document.toString());
		SchemaFactory factory = SchemaFactory
				.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
		Source schemaFile = new StreamSource(xsdFile);
		// log.info("Xsd=" + schemaFile.toString());
		Schema schema = factory.newSchema(schemaFile);
		Validator validator = schema.newValidator();
		try
		{
			validator.validate(new DOMSource(document));
			isValid = true;
		}
		catch (SAXException e)
		{
			log.error("Failed to validate xml", e);
			throw e;
		}
		if (log.isInfoEnabled())
		{
			log.info("IsValid = " + isValid);
		}
		return isValid;
	}

}
