/*
 * JBoss, Home of Professional Open Source Copyright 2008, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.security.opensso;

import java.io.Serializable;
import java.security.Principal;

/**
 * A SSOPrincipal represents a caller that has been authenticated by
 * a Single Sign On system.
 * <p/>
 * 
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class SSOPrincipal implements Principal, Serializable
{
	private static final long serialVersionUID = 1L;
	
	private final String ssoToken;

	public SSOPrincipal(final String ssoToken)
	{
		if ( ssoToken == null )
			throw new NullPointerException("ssoToken argument must not be null");
		
		this.ssoToken = ssoToken;
	}

	/**
	 * Returns the name of this Principal.
	 * @return String - will always return "SSOToken".
	 */
	public String getName()
	{
		return "SSOToken";
	}
	
	/**
	 * Returns a String representation of the token used to identify
	 * an authenticated user/system.
	 * 
	 * @return String - the SSO systems token.
	 */
	public String getToken()
	{
		return ssoToken;
	}
	
	public String toString()
	{
		return "SSOToken [ssoToken=" + ssoToken + "]";
	}

	@Override
	public boolean equals(Object obj)
	{
		if (this == obj)
			return true;
		if(!(obj instanceof SSOPrincipal))
			return false;
		
		SSOPrincipal other = (SSOPrincipal) obj; 
		return this.ssoToken.equals(other.ssoToken);
	}

	@Override
	public int hashCode()
	{
		int hash = 17;
		hash = 31 * hash + ssoToken == null ? 0 : ssoToken.hashCode();
		return hash;
	}

}
