/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.msg.jms;

import javax.jms.Connection;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageProducer;
import javax.jms.Session;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.jbpm.JbpmContext;
import org.jbpm.JbpmException;
import org.jbpm.db.JobSession;
import org.jbpm.graph.exe.Token;
import org.jbpm.job.Job;
import org.jbpm.msg.MessageService;
import org.jbpm.svc.Services;
import org.jbpm.taskmgmt.exe.TaskInstance;
import org.jbpm.tx.TxService;

public class JmsMessageService implements MessageService {

  private static final long serialVersionUID = 1L;

  private static final Log log = LogFactory.getLog(JmsMessageService.class);

  final JobSession jobSession;

  final Connection connection;
  final Session session;
  final MessageProducer messageProducer;

  final boolean isCommitEnabled;

  /**
   * @deprecated use {@link #JmsMessageService(JmsMessageServiceFactory)} instead
   */
  public JmsMessageService(Connection connection, Destination destination, boolean isCommitEnabled)
      throws JMSException {
    JbpmContext jbpmContext = JbpmContext.getCurrentJbpmContext();
    if (jbpmContext == null) throw new JbpmException("no active jbpm context");
    jobSession = jbpmContext.getJobSession();

    this.connection = connection;

    if (isCommitEnabled) {
      session = connection.createSession(true, Session.SESSION_TRANSACTED);
      this.isCommitEnabled = true;
    }
    else {
      session = connection.createSession(false, Session.AUTO_ACKNOWLEDGE);
      this.isCommitEnabled = false;
    }

    messageProducer = session.createProducer(destination);
  }

  public JmsMessageService(JmsMessageServiceFactory factory) throws JMSException {
    this(factory.getConnectionFactory().createConnection(), factory.getDestination(),
        factory.isCommitEnabled());
  }

  public void send(Job job) {
    jobSession.saveJob(job);

    try {
      Message message = session.createMessage();
      message.setLongProperty("jobId", job.getId());

      Token token = job.getToken();
      if (token != null) {
        message.setLongProperty("tokenId", token.getId());
        message.setLongProperty("processInstanceId", job.getProcessInstance().getId());
      }

      TaskInstance taskInstance = job.getTaskInstance();
      if (taskInstance != null) {
        message.setLongProperty("taskInstanceId", taskInstance.getId());
      }

      modifyMessage(message, job);
      messageProducer.send(message);
    }
    catch (JMSException e) {
      throw new JbpmException("could not send jms message", e);
    }
  }

  /**
   * Hook to modify the message, e.g. adding extra properties to the header required by the own
   * application. One possible use case is to rescue the actor id over the "JMS" intermezzo of
   * asynchronous continuations.
   */
  protected void modifyMessage(Message message, Job job) throws JMSException {
  }

  public void close() {
    try {
      messageProducer.close();
    }
    catch (JMSException e) {
      log.warn("could not close message producer", e);
    }

    JMSException commitException = null;
    if (isCommitEnabled) {
      TxService txService = (TxService) Services.getCurrentService(Services.SERVICENAME_TX);
      try {
        if (txService.isRollbackOnly()) {
          session.rollback();
        }
        else {
          session.commit();
        }
      }
      catch (JMSException e) {
        commitException = e;
      }
    }

    try {
      session.close();
    }
    catch (JMSException e) {
      log.warn("could not close jms session", e);
    }

    try {
      connection.close();
    }
    catch (JMSException e) {
      log.warn("could not close jms connection", e);
    }

    if (commitException != null) {
      throw new JbpmException("could not commit jms session", commitException);
    }
  }

  public Session getSession() {
    return session;
  }

  protected MessageProducer getMessageProducer() throws JMSException {
    return messageProducer;
  }
}
