package org.jbpm.jbpm2375;

import org.jbpm.db.AbstractDbTestCase;
import org.jbpm.graph.def.ActionHandler;
import org.jbpm.graph.def.ProcessDefinition;
import org.jbpm.graph.exe.ExecutionContext;
import org.jbpm.graph.exe.ProcessInstance;

/**
 * Test if the JobExecutorThread recovers from an Error
 * 
 * @see <a href="https://jira.jboss.org/jira/browse/JBPM-2357">JBPM-2357</a>
 * @author mputz@redhat.com
 * @since 30-Jun-2009
 */
public class JBPM2375Test extends AbstractDbTestCase {

  private static final int TEST_TIMEOUT = 60 * 1000;
  private static boolean throwError;

  private ProcessDefinition processDefinition;

  // a process definition with two timers moving the token forward
  // the second state has an action associated with the node-enter event,
  // which can simulate an Error condition by throwing a NoClassDefFoundError
  private static final String PROCESS_DEFINITION = "<process-definition name='jbpm2375-test'>" +
      "  <start-state name='start'>" +
      "    <transition to='state1' name='to_state1'/>" +
      "  </start-state>" +
      "  <state name='state1'>" +
      "    <timer name='moveToNextStateAfter1second' duedate='1 second' transition='to_state2'/>" +
      "    <transition to='state2' name='to_state2'/>" +
      "  </state>" +
      "  <state name='state2'>" +
      "    <timer name='moveToEndAfter1second' duedate='1 second' transition='to_end'/>" +
      "    <event type='node-enter'>" +
      "      <action name='exceptionTest' class='" +
      TimerExceptionAction.class.getName() +
      "'>" +
      "      </action>" +
      "    </event>" +
      "    <transition to='end' name='to_end'/>" +
      "  </state>" +
      "  <end-state name='end' />" +
      "</process-definition>";

  protected void setUp() throws Exception {
    super.setUp();
    jbpmConfiguration.getJobExecutor().setLockMonitorInterval(TEST_TIMEOUT / 6);

    processDefinition = ProcessDefinition.parseXmlString(PROCESS_DEFINITION);
    jbpmContext.deployProcessDefinition(processDefinition);

    newTransaction();
  }

  protected void tearDown() throws Exception {
    graphSession.deleteProcessDefinition(processDefinition.getId());

    jbpmConfiguration.getJobExecutor().setLockMonitorInterval(60000);
    super.tearDown();
  }

  // check if the process ends correctly if no Error is thrown
  public void testTimerWithoutErrorAction() {
    throwError = false;
    runTimerErrorAction();
  }

  // check if the process ends correctly if an Error is thrown in the ActionHandler
  public void testTimerWithErrorAction() {
    throwError = true;
    runTimerErrorAction();
  }

  private void runTimerErrorAction() {
    // kick off process instance
    ProcessInstance processInstance = new ProcessInstance(processDefinition);
    processInstance.signal();
    jbpmContext.save(processInstance);

    processJobs(TEST_TIMEOUT);

    processInstance = jbpmContext.loadProcessInstance(processInstance.getId());
    assert processInstance.hasEnded() : processInstance;
  }

  public static class TimerExceptionAction implements ActionHandler {
    private static final long serialVersionUID = 1L;

    public void execute(ExecutionContext executionContext) throws Exception {
      if (throwError) {
        throwError = false;
        throw new NoClassDefFoundError("org.jbpm.no.such.Class");
      }
    }
  }
}
