/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.services.soapui;

import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.StringWriter;
import java.math.BigDecimal;
import java.util.*;

import javax.xml.transform.stream.StreamResult;

import junit.framework.TestCase;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.services.soapui.orderprocessing.Order;
import org.jboss.soa.esb.services.soapui.orderprocessing.LineItem;
import org.jboss.soa.esb.services.soapui.orderprocessing.ProcessOrderRequest;
import org.jboss.soa.esb.http.HttpClientFactory;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

/**
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class SoapUIClientServiceMBeanUnitTest extends TestCase {

    private static final String WSDL_LOCATAION = "src/test/java/org/jboss/soa/esb/services/soapui";
    private Properties properties;

    protected void setUp() throws Exception {
        properties = new Properties();
    }

    public void test_no_collections() throws IOException, SAXException, ConfigurationException {
        File wsdlFile = new File(WSDL_LOCATAION + "/RetailerCallback.wsdl");
        SoapUIClientService mbean = new SoapUIClientService();
        Map params = new HashMap();

        properties.setProperty(HttpClientFactory.TARGET_HOST_URL, wsdlFile.toURI().toString());
        params.put("salesOrderNotification", new OrderNotification());
        String message = mbean.buildRequest(wsdlFile.toURI().toString(), "SendSalesOrderNotification", params, properties, null, null);
        assertTrue("Generated SOAP message not as expected. See expected_01.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("expected_01.xml"), new ByteArrayInputStream(message.getBytes())));
        assertEquals("http://localhost:18080/active-bpel/services/RetailerCallback", mbean.getEndpoint(wsdlFile.toURI().toString(), properties));
    }

    public void test_has_collections_01() throws IOException, SAXException, ConfigurationException {
        File wsdlFile = new File(WSDL_LOCATAION + "/BPELRetailer.wsdl");
        String soapNs = null;
        SoapUIClientService mbean = new SoapUIClientService();
        Map params = new HashMap();
        CustomerOrder1 order1 = new CustomerOrder1();
        CustomerOrder2 order2 = new CustomerOrder2();
        List<OrderItem> items1 = new ArrayList<OrderItem>();
        OrderItem[] items2 = new OrderItem[4];

        params.put("dumpSOAP", null);
        
        properties.setProperty(HttpClientFactory.TARGET_HOST_URL, wsdlFile.toURI().toString());

        addOrderItems(items1);
        order1.items = items1;
        items1.toArray(items2);
        order2.items = items2;

        params.put("customerOrder", order1);
        String message = mbean.buildRequest(wsdlFile.toURI().toString(), "SubmitOrder", params, properties, null, soapNs);
        assertTrue("Generated SOAP message not as expected. See expected_02.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("expected_02.xml"), new ByteArrayInputStream(message.getBytes())));

        items1.remove(0);items1.remove(0);items1.remove(0);
        message = mbean.buildRequest(wsdlFile.toURI().toString(), "SubmitOrder", params, properties, null, soapNs);
        assertTrue("Generated SOAP message not as expected. See expected_03.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("expected_03.xml"), new ByteArrayInputStream(message.getBytes())));

        params.put("customerOrder", order2);
        message = mbean.buildRequest(wsdlFile.toURI().toString(), "SubmitOrder", params, properties, null, soapNs);
        assertTrue("Generated SOAP message not as expected. See expected_02.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("expected_02.xml"), new ByteArrayInputStream(message.getBytes())));

        items2 = new OrderItem[] {new OrderItem(4, "item4", 4, new BigDecimal(4.00), 4)};        
        order2.items = items2;
        params.put("customerOrder", order2);
        message = mbean.buildRequest(wsdlFile.toURI().toString(), "SubmitOrder", params, properties, null, soapNs);
        assertTrue("Generated SOAP message not as expected. See expected_03.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("expected_03.xml"), new ByteArrayInputStream(message.getBytes())));

        params.clear();
        params.put("customerOrder.items[0].partNumber", 4);
        params.put("customerOrder.items[0].description", "item4");
        params.put("customerOrder.items[0].quantity", 4);
        params.put("customerOrder.items[0].price", 4);
        params.put("customerOrder.items[0].extensionAmount", 4);
        message = mbean.buildRequest(wsdlFile.toURI().toString(), "SubmitOrder", params, properties, null, soapNs);
        assertTrue("Generated SOAP message not as expected. See expected_03.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("expected_03.xml"), new ByteArrayInputStream(message.getBytes())));
    }

    public void test_has_collections_02() throws IOException, SAXException, ConfigurationException {
        File wsdlFile = new File(WSDL_LOCATAION + "/orderprocessing/OrderProcessing.wsdl");
        SoapUIClientService mbean = new SoapUIClientService();
        Map params = new HashMap();
        ArrayList<LineItem> lineItems = new ArrayList<LineItem>();

        params.put("dumpSOAP", null);

        properties.setProperty(HttpClientFactory.TARGET_HOST_URL, wsdlFile.toURI().toString());

        Order order = new Order();
        ProcessOrderRequest request = new ProcessOrderRequest(order);
        order.setId(123L);
        order.setShipTo("Skeagh Bridge");
        order.setLineItems(lineItems);

        LineItem lineItem;

        lineItem = new LineItem();
        lineItems.add(lineItem);
        lineItem.setId(456L);
        lineItem.setName("item1");
        lineItem.setPrice(10.99f);

        lineItem = new LineItem();
        lineItems.add(lineItem);
        lineItem.setId(890L);
        lineItem.setName("item2");
        lineItem.setPrice(12.11f);

        lineItem = new LineItem();
        lineItems.add(lineItem);
        lineItem.setId(321L);
        lineItem.setName("item3");
        lineItem.setPrice(76.34f);

        params.put("processOrder", request);
        String message = mbean.buildRequest(wsdlFile.toURI().toString(), "processOrder", params, properties, null, null);
        assertTrue("Generated SOAP message not as expected. See orderprocessing/expected_01.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("orderprocessing/expected_01.xml"), new ByteArrayInputStream(message.getBytes())));
    }

    public void test_has_collections_02_JBESB_1329() throws IOException, SAXException, ConfigurationException {
        File wsdlFile = new File(WSDL_LOCATAION + "/orderprocessing/OrderProcessing_JBESB-1329.wsdl");
        SoapUIClientService mbean = new SoapUIClientService();
        Map params = new HashMap();
        ArrayList<LineItem> lineItems = new ArrayList<LineItem>();

        properties.setProperty(HttpClientFactory.TARGET_HOST_URL, wsdlFile.toURI().toString());

        Order order = new Order();
        ProcessOrderRequest request = new ProcessOrderRequest(order);
        order.setId(123L);
        order.setShipTo("Skeagh Bridge");
        order.setLineItems(lineItems);

        LineItem lineItem;

        lineItem = new LineItem();
        lineItems.add(lineItem);
        lineItem.setId(456L);
        lineItem.setName("item1");
        lineItem.setPrice(10.99f);

        lineItem = new LineItem();
        lineItems.add(lineItem);
        lineItem.setId(890L);
        lineItem.setName("item2");
        lineItem.setPrice(12.11f);

        lineItem = new LineItem();
        lineItems.add(lineItem);
        lineItem.setId(321L);
        lineItem.setName("item3");
        lineItem.setPrice(76.34f);

        params.put("processOrder", request);
        String message = mbean.buildRequest(wsdlFile.toURI().toString(), "processOrder", params, properties, null, null);
        assertTrue("Generated SOAP message not as expected. See orderprocessing/expected_01_JBESB-1329.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("orderprocessing/expected_01_JBESB-1329.xml"), new ByteArrayInputStream(message.getBytes())));
    }

    public void test_has_collections_03() throws IOException, SAXException, ConfigurationException {
        File wsdlFile = new File(WSDL_LOCATAION + "/orderprocessing/OrderProcessing.wsdl");
        SoapUIClientService mbean = new SoapUIClientService();
        Map params = new HashMap();
        ArrayList<LineItem> lineItems = new ArrayList<LineItem>();

        properties.setProperty(HttpClientFactory.TARGET_HOST_URL, wsdlFile.toURI().toString());

        Order order = new Order();
        ProcessOrderRequest request = new ProcessOrderRequest(order);
        order.setId(123L);
        order.setShipTo("Skeagh Bridge");
        order.setLineItems(lineItems);

        // No line Items....

        params.put("processOrder", request);
        String message = mbean.buildRequest(wsdlFile.toURI().toString(), "processOrder", params, properties, null, null);
        assertTrue("Generated SOAP message not as expected. See orderprocessing/expected_02.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("orderprocessing/expected_02.xml"), new ByteArrayInputStream(message.getBytes())));
    }

    public void test_has_collections_03_JBESB_1329() throws IOException, SAXException, ConfigurationException {
        File wsdlFile = new File(WSDL_LOCATAION + "/orderprocessing/OrderProcessing_JBESB-1329.wsdl");
        SoapUIClientService mbean = new SoapUIClientService();
        Map params = new HashMap();
        ArrayList<LineItem> lineItems = new ArrayList<LineItem>();

        properties.setProperty(HttpClientFactory.TARGET_HOST_URL, wsdlFile.toURI().toString());

        Order order = new Order();
        ProcessOrderRequest request = new ProcessOrderRequest(order);
        order.setId(123L);
        order.setShipTo("Skeagh Bridge");
        order.setLineItems(lineItems);

        // No line Items....

        params.put("processOrder", request);
        String message = mbean.buildRequest(wsdlFile.toURI().toString(), "processOrder", params, properties, null, null);
        assertTrue("Generated SOAP message not as expected. See orderprocessing/expected_02_JBESB-1329.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("orderprocessing/expected_02_JBESB-1329.xml"), new ByteArrayInputStream(message.getBytes())));
    }

    public void test_has_collections_04() throws IOException, SAXException, ConfigurationException {
        File wsdlFile = new File(WSDL_LOCATAION + "/orderprocessing/OrderProcessing.wsdl");
        SoapUIClientService mbean = new SoapUIClientService();
        Map params = new HashMap();

        properties.setProperty(HttpClientFactory.TARGET_HOST_URL, wsdlFile.toURI().toString());

        params.put("processOrder.order.id", 123L);
        params.put("processOrder.order.shipTo", "Skeagh Bridge");

        params.put("processOrder.order.lineItems[0].id", 456L);
        params.put("processOrder.order.lineItems[0].name", "item1");
        params.put("processOrder.order.lineItems[0].price", 10.99f);
        params.put("processOrder.order.lineItems[1].id", 890L);
        params.put("processOrder.order.lineItems[1].name", "item2");
        params.put("processOrder.order.lineItems[1].price", 12.11f);
        params.put("processOrder.order.lineItems[2].id", 321L);
        params.put("processOrder.order.lineItems[2].name", "item3");
        params.put("processOrder.order.lineItems[2].price", 76.34f);

        String message = mbean.buildRequest(wsdlFile.toURI().toString(), "processOrder", params, properties, null, null);
        assertTrue("Generated SOAP message not as expected. See orderprocessing/expected_01.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("orderprocessing/expected_01.xml"), new ByteArrayInputStream(message.getBytes())));
    }

    public void test_has_collections_04_JBESB_1329() throws IOException, SAXException, ConfigurationException {
        File wsdlFile = new File(WSDL_LOCATAION + "/orderprocessing/OrderProcessing_JBESB-1329.wsdl");
        SoapUIClientService mbean = new SoapUIClientService();
        Map params = new HashMap();

        properties.setProperty(HttpClientFactory.TARGET_HOST_URL, wsdlFile.toURI().toString());

        params.put("processOrder.order.id", 123L);
        params.put("processOrder.order.shipTo", "Skeagh Bridge");

        params.put("processOrder.order.lineItems[0].id", 456L);
        params.put("processOrder.order.lineItems[0].name", "item1");
        params.put("processOrder.order.lineItems[0].price", 10.99f);
        params.put("processOrder.order.lineItems[1].id", 890L);
        params.put("processOrder.order.lineItems[1].name", "item2");
        params.put("processOrder.order.lineItems[1].price", 12.11f);
        params.put("processOrder.order.lineItems[2].id", 321L);
        params.put("processOrder.order.lineItems[2].name", "item3");
        params.put("processOrder.order.lineItems[2].price", 76.34f);

        String message = mbean.buildRequest(wsdlFile.toURI().toString(), "processOrder", params, properties, null, null);
        assertTrue("Generated SOAP message not as expected. See orderprocessing/expected_01_JBESB-1329.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("orderprocessing/expected_01_JBESB-1329.xml"), new ByteArrayInputStream(message.getBytes())));
    }

    public void test_has_collections_05() throws IOException, SAXException, ConfigurationException {
        File wsdlFile = new File(WSDL_LOCATAION + "/orderprocessing/OrderProcessing.wsdl");
        SoapUIClientService mbean = new SoapUIClientService();
        Map params = new HashMap();

        properties.setProperty(HttpClientFactory.TARGET_HOST_URL, wsdlFile.toURI().toString());

        params.put("processOrder.order.id", 123L);
        params.put("processOrder.order.shipTo", "Skeagh Bridge");

        // No line Items....

        String message = mbean.buildRequest(wsdlFile.toURI().toString(), "processOrder", params, properties, null, null);
        assertTrue("Generated SOAP message not as expected. See orderprocessing/expected_02.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("orderprocessing/expected_02.xml"), new ByteArrayInputStream(message.getBytes())));
    }

    public void test_smooks_transform() throws IOException, SAXException, ConfigurationException {
        File wsdlFile = new File(WSDL_LOCATAION + "/RetailerCallback.wsdl");
        SoapUIClientService mbean = new SoapUIClientService();
        Map params = new HashMap();

        properties.setProperty(HttpClientFactory.TARGET_HOST_URL, wsdlFile.toURI().toString());
        params.put("salesOrderNotification", new OrderNotification());
        String message = mbean.buildRequest(wsdlFile.toURI().toString(), "SendSalesOrderNotification", params, properties, StreamUtils.readStreamString(getClass().getResourceAsStream("sendNotificationTransform.xml"), "UTF-8"), null);
        assertTrue("Generated SOAP message not as expected. See expected_04.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("expected_04.xml"), new ByteArrayInputStream(message.getBytes())));
        assertEquals("http://localhost:18080/active-bpel/services/RetailerCallback", mbean.getEndpoint(wsdlFile.toURI().toString(), properties));
    }

    public void test_BuildResponse() throws Exception {    
    	File wsdlFile = new File(WSDL_LOCATAION + "/helloworld.wsdl");
    	properties.setProperty(HttpClientFactory.TARGET_HOST_URL, wsdlFile.toURI().toString());
    	SoapUIClientService mbean = new SoapUIClientService();
    	Map<Object, Object> paras = new HashMap<Object, Object>();
    	paras.put("sayHiReponse.arg0", "response");
    	String str = mbean.buildResponse(wsdlFile.toURL().toString(), "HelloWorldPubServiceOp", paras, properties, null, null);
    	assertTrue("Failed to generate correct soap response", str.indexOf("<say:arg0>response</say:arg0>") > -1);
    }
    
    public void test_BuildFault() throws Exception {    
    	File wsdlFile = new File(WSDL_LOCATAION + "/helloworld.wsdl");
    	properties.setProperty(HttpClientFactory.TARGET_HOST_URL, wsdlFile.toURI().toString());
    	SoapUIClientService mbean = new SoapUIClientService();
    	Map<Object, Object> paras = new HashMap<Object, Object>();
    	paras.put("Fault.detail.sayFault.code", "test");
    	String str = mbean.buildFault(wsdlFile.toURL().toString(), "HelloWorldPubServiceOp", "HelloWorldPubServiceFault1", paras, properties, null, null);
    	assertTrue("Failed to generate correct soap fault message", str.indexOf("<say:code>test</say:code>") > -1);
    }
    
    public void test_has_attributes() throws IOException, SAXException, ConfigurationException {
        File wsdlFile = new File(WSDL_LOCATAION + "/RetailerCallback.wsdl");
        SoapUIClientService mbean = new SoapUIClientService();
        Map<String, CancelOrder> params = new HashMap<String, CancelOrder>();

        properties.setProperty(HttpClientFactory.TARGET_HOST_URL, wsdlFile.toURI().toString());
        CancelOrder cancelorder = new CancelOrder();
        cancelorder.orderInquiry.inquiryId = "X2343443";
        cancelorder.orderInquiry.customerNumber = "12345";
        cancelorder.orderInquiry.poNumber = "asdsadfdfd";

        params.put("cancelOrder", cancelorder);

        String message = mbean.buildRequest(wsdlFile.toURI().toString(), "CancelOrder", params, properties, null, null);
        assertTrue("Generated SOAP message not as expected. See expected_05.xml.  Generated message: \n" + message, compareCharStreams(getClass().getResourceAsStream("expected_05.xml"), new ByteArrayInputStream(message.getBytes())));
        assertEquals("http://localhost:18080/active-bpel/services/RetailerCallback", mbean.getEndpoint(wsdlFile.toURI().toString(), properties));
    }


    
    private void addOrderItems(List<OrderItem> items) {
        items.add(new OrderItem(1, "item1", 1, new BigDecimal(1.00), 1));
        items.add(new OrderItem(2, "item2", 2, new BigDecimal(2.00), 2));
        items.add(new OrderItem(3, "item3", 3, new BigDecimal(3.00), 3));
        items.add(new OrderItem(4, "item4", 4, new BigDecimal(4.00), 4));
    }

    private class OrderNotification {
        public String orderNumber = "12345";
    }

    private class CustomerOrder1 {
        public List<OrderItem> items;
    }

    private class CustomerOrder2 {
        public OrderItem[] items;
    }

    private class OrderItem {
        public long partNumber;
        public String description;
        public int quantity;
        public BigDecimal price;
        public int extensionAmount;

        public OrderItem(long partNumber, String description, int quantity, BigDecimal price, int extensionAmount) {
            this.partNumber = partNumber;
            this.description = description;
            this.quantity = quantity;
            this.price = price;
            this.extensionAmount = extensionAmount;
        }
    }

    // Lifted from milyn commons
    public static boolean compareCharStreams(InputStream s1, InputStream s2) {
        try {
            final String xml1 = trimLines(s1).toString() ;
            final String xml2 = trimLines(s2).toString() ;
            
            final Document doc1 = YADOMUtil.parse(xml1) ;
            final Document doc2 = YADOMUtil.parse(xml2) ;
            
            final StringWriter writer1 = new StringWriter() ;
            final StringWriter writer2 = new StringWriter() ;
            YADOMUtil.serialize(doc1, new StreamResult(writer1)) ;
            YADOMUtil.serialize(doc2, new StreamResult(writer2)) ;
            return (writer1.toString().equals(writer2.toString())) ;
        } catch (IOException e) {
            // fail the comparison
        } catch (SAXException e) {
            // fail the comparison
        } catch (ConfigurationException e) {
            // fail the comparison
        }

        return false;
    }
    public static StringBuffer trimLines(InputStream charStream) throws IOException {
        StringBuffer stringBuf = new StringBuffer();
        BufferedReader reader = new BufferedReader(new InputStreamReader(charStream));
        String line;

        while((line = reader.readLine()) != null) {
            stringBuf.append(line.trim());
        }

        return stringBuf;
    }
    
    public static void main(final String[] args)
    {
        final InputStream expectedIS = SoapUIClientServiceMBeanUnitTest.class.getResourceAsStream("expected_02.xml") ;
        final InputStream outputIS = SoapUIClientServiceMBeanUnitTest.class.getResourceAsStream("output.xml") ;
        
        System.out.println("compare returns: " + compareCharStreams(expectedIS, outputIS));
    }    
    
    private class CancelOrder {
       public OrderInquiry orderInquiry = new OrderInquiry();
    }
   
    private class OrderInquiry {
       public String customerNumber;
       public String poNumber;
       public String inquiryId;
    }
   

}
