/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.soap;

import java.net.URI;
import java.net.URISyntaxException;

import org.apache.log4j.Logger;
import org.milyn.SmooksException;
import org.milyn.container.ExecutionContext;
import org.milyn.delivery.dom.DOMVisitAfter;
import org.milyn.xml.DomUtils;
import org.w3c.dom.Element;

/**
 * Smooks transformer for transforming the WSDL for ESB endpoints.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class WsdlEndpointTransformer implements DOMVisitAfter {

    private static Logger logger = Logger.getLogger(WsdlEndpointTransformer.class);

    public static final String REWRITE_ENDPOINT_URL = "rewriteEndpointUrl";
    public static final String ENDPOINT_URI = "endpointURI";
    public static final String TARGET_CAT = "targetServiceCat";
    public static final String TARGET_NAME = "targetServiceName";
    public static final String TARGET_PROTOCOL = "targetProtocol";

    public void visitAfter(Element element, ExecutionContext executionContext) throws SmooksException {
        String elementName = DomUtils.getName(element);
        boolean rewriteEndpointUrl = (Boolean) executionContext.getAttribute(WsdlEndpointTransformer.REWRITE_ENDPOINT_URL);
        URI endpointURI = (URI) executionContext.getAttribute(ENDPOINT_URI);

        if(!rewriteEndpointUrl && endpointURI.getScheme().startsWith("http")) {
            // Don't perform any rewriting ops...
            return;
        }

        if(elementName.equals("import")) {
            transformImport(element, executionContext);
        } else if(elementName.equals("binding")) {
            transformBinding(element, executionContext);
        } else if(elementName.equals("address")) {
            transformAddress(element, executionContext);
        }
    }

    private void transformImport(Element element, ExecutionContext executionContext) {
        String location = DomUtils.getAttributeValue(element, "location");

        if(location != null) {
            try {
            	// used just to trigger a URISyntaxException so we can break out fast
                @SuppressWarnings("unused")
				URI locationURI = new URI(location);
                
                String targetServiceCat = (String) executionContext.getAttribute(WsdlEndpointTransformer.TARGET_CAT);
                String targetServiceName = (String) executionContext.getAttribute(WsdlEndpointTransformer.TARGET_NAME);
                String targetProtocol = (String) executionContext.getAttribute(WsdlEndpointTransformer.TARGET_PROTOCOL);
                
                StringBuilder locationBuilder = new StringBuilder(location);
                appendParam(locationBuilder, "serviceCat", targetServiceCat);
                appendParam(locationBuilder, "serviceName", targetServiceName);
                appendParam(locationBuilder, "protocol", targetProtocol);
                
                element.setAttribute( "location", locationBuilder.toString() );
            } catch (URISyntaxException e) {
                logger.info("Not processing WSDL import '" + location + "'.  Invalid URI.");
            }
        }
    }
    
    private void appendParam(StringBuilder builder, String name, String value) throws URISyntaxException {
    	if (value != null && value.length() > 0) {
    		String qPrefix = "?" + name + "=";
    		String aPrefix = "&" + name + "=";
    		String location = builder.toString();
    		if (!location.contains(qPrefix) && !location.contains(aPrefix)) {
    			builder.append(new URI(location).getQuery() == null ? qPrefix : aPrefix);
    			builder.append(value);
    		}
    	}
    }

    private void transformBinding(Element element, ExecutionContext executionContext) {
        URI endpointURI = (URI) executionContext.getAttribute(ENDPOINT_URI);
        if(endpointURI != null) {
            element.setAttribute("transport", "http://schemas.xmlsoap.org/soap/" + endpointURI.getScheme());
        }
    }

    private void transformAddress(Element element, ExecutionContext executionContext) {
        URI endpointURI = (URI) executionContext.getAttribute(ENDPOINT_URI);
        if(endpointURI != null) {
            // Make sure there's a path (if only a single slash).  JBossWS client code will
            // fail because of a feature of JBossRemoting.  See http://jira.jboss.com/jira/browse/JBWS-1802
            if(endpointURI.getPath() == null || endpointURI.getPath().equals("")) {
                try {
                    endpointURI = new URI(endpointURI.getScheme(), endpointURI.getUserInfo(),
                            endpointURI.getHost(), endpointURI.getPort(), "/", endpointURI.getQuery(), endpointURI.getFragment());
                } catch (URISyntaxException e) {
                    logger.error("Failed to properly format endpoint location URL '" + endpointURI.toString() + "'.  Failed to add single slash character for path.  This may cause JBossWS Client code to fail.");
                }
            }
            
            element.setAttribute("location", endpointURI.toString());
        }
    }
}
