/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.actions.soap;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.Properties;

import org.apache.commons.httpclient.HostConfiguration;
import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethodBase;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.log4j.Logger;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.routing.http.HttpMethodFactory;
import org.jboss.soa.esb.http.HttpClientFactory;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * {@link HttpClientFactory} based WSLD loader.
 * <p/>
 * We use this in order to support different authentication modes that
 * may be required for WSDL lookup.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class RemoteWsdlLoader {

    private static Logger logger = Logger.getLogger(RemoteWsdlLoader.class);
    private HttpClient httpClient;
    private HostConfiguration hostConfig;

    public RemoteWsdlLoader(Properties httpClientProps) throws ConfigurationException {
        httpClient = HttpClientFactory.createHttpClient(httpClientProps);
        hostConfig = httpClient.getHostConfiguration();
        httpClient.setHostConfiguration(new HostConfiguration());
    }

    public InputStream load(String url) throws IOException {
        if (httpClient == null) {
            throw new IOException("RemoteWsdlLoader already shutdown") ;
        }
        HttpMethodBase httpMethod;

        // short-circuits
        if(url.startsWith("file")) {
            return new URL(url).openStream();
        } else if(url.startsWith("classpath://")) {
        	return ClassUtil.getResource(url.substring(12, url.length()), getClass()).openStream();
        }
        
        try {
			HttpMethodFactory methodFactory = HttpMethodFactory.Factory.getInstance("GET", null, new URL(url));
			httpMethod = methodFactory.getInstance(null);
		} catch (ConfigurationException ce) {
			// should never happen with "GET" method since ConfigTree isn't used with GET - only POST
			throw (IOException)(new IOException(ce.getMessage()).initCause(ce));
		}

        // Authentication is not being overridden on the method.  It needs
        // to be present on the supplied HttpClient instance!
        httpMethod.setDoAuthentication(true);

        try {
        	int result = httpClient.executeMethod(hostConfig, httpMethod);

            if(result != HttpStatus.SC_OK) {
                if(result < 200 || result > 299) {
                    throw new HttpException("Received status code '" + result + "' on WSDL HTTP (GET) request: '" + url + "'.");
                } else {
                    logger.warn("Received status code '" + result + "' on WSDL HTTP (GET) request: '" + url + "'.");
                }
            }

            return new ByteArrayInputStream(httpMethod.getResponseBody());
        } finally {
            httpMethod.releaseConnection();
        }
    }
    
    public void shutdown() {
        HttpClientFactory.shutdown(httpClient);
        httpClient = null ;
    }
}

