/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.actions.converters;


import java.net.URISyntaxException;
import java.util.Map;

import junit.framework.TestCase;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionLifecycleException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.MessagePayloadProxy;
import org.jboss.soa.esb.message.format.MessageFactory;

/**
 * SmooksTransformer unit tests.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 * @since Version 4.0
 */
public class SmooksTransformerUnitTest extends TestCase {

    public void test_bad_config() {
        ConfigTree properties = new ConfigTree("name");

        properties.setAttribute(SmooksTransformer.RESOURCE_CONFIG, "classpath:/smooks-conf-01.xml");
        // Should get exceptions where message type configs are specified but empty...
        properties.setAttribute(SmooksTransformer.FROM_TYPE, " ");
        assertConfigException(properties, "Empty '" + SmooksTransformer.FROM_TYPE + "' config attribute supplied.");
        properties.setAttribute(SmooksTransformer.FROM_TYPE, "x");
        properties.setAttribute(SmooksTransformer.TO_TYPE, " ");
        assertConfigException(properties, "Empty '" + SmooksTransformer.TO_TYPE + "' config attribute supplied.");
    }

    public void test_trans() throws ActionProcessingException, ConfigurationException, URISyntaxException, ActionLifecycleException {
        String stringMessage;
        String transRes;

        // Very basic test!  Just illustrates profile based resource selection.
        // Read the smooks-test.cdrl config file in this package!! 

        // Initialise the acme order message...
        stringMessage = "<a><ddd>value</ddd></a>";

        // Transform the order message going to "AcmePartner1"...
        transRes = transform(stringMessage, "Acme-Order-XML", "Acme", "AcmePartner1", "Partner1-Order-XML");
        assertEquals("<x><b>value</b></x>", transRes);

        // Transform the order message going to "AcmePartner2"...
        transRes = transform(stringMessage, "Acme-Order-XML", "Acme", "AcmePartner2", "Partner2-Order-XML");
        assertEquals("<x><c>value</c></x>", transRes);
    }

    public void test_javabeans() throws ConfigurationException, ActionLifecycleException, ActionProcessingException {
        ConfigTree properties = new ConfigTree("smooks-config");
        SmooksTransformer transformer;
        Message message;

        properties.setAttribute(SmooksTransformer.RESOURCE_CONFIG, "/org/jboss/soa/esb/actions/converters/smooks-conf.xml");
        properties.setAttribute(SmooksTransformer.JAVA_OUTPUT, "ObjectMap");
        transformer = new SmooksTransformer(properties);
        transformer.initialise();

        // Perform the transformation by setting the payload on the task object...
        message = MessageFactory.getInstance().getMessage();
        message.getBody().add("<x xprop='xval' />");

        assertNull(message.getBody().get("orderHeader"));
        message = transformer.process(message);
        assertNotNull(message.getBody().get("orderHeader"));
        assertTrue(message.getBody().get("orderHeader") instanceof XClass);
        assertNotNull(message.getBody().get("ObjectMap"));
        assertTrue(message.getBody().get("ObjectMap") instanceof Map);
    }

    private String transform(String stringMessage, String fromType, String from, String to, String toType) throws ActionProcessingException, ConfigurationException, URISyntaxException, ActionLifecycleException {
        ConfigTree properties = createConfig01(from, fromType, to, toType);
        SmooksTransformer transformer = new SmooksTransformer(properties);
        Message message;

        transformer.initialise();

        // Perform the transformation by setting the payload on the task object...
        message = MessageFactory.getInstance().getMessage();
        message.getBody().add(stringMessage);
        message = transformer.process(message);
        String resultThroughTaskObj = (String) message.getBody().get();
        assertEquals(resultThroughTaskObj, message.getBody().get());

        // Perform the transformation by setting the payload on the body as bytes...
        message = createNewMessage(fromType, from, to, toType);
        message.getBody().add(stringMessage.getBytes());
        message = transformer.process(message);
        assertEquals(resultThroughTaskObj, message.getBody().get());

        // Perform the transformation by setting/getting the payload on the default body location...
        message = MessageFactory.getInstance().getMessage();
        message.getBody().add(stringMessage);
        message = transformer.process(message);
        assertEquals(resultThroughTaskObj, message.getBody().get());

        // Perform the transformation by setting the payload input and output body locations...
        properties.setAttribute(MessagePayloadProxy.GET_PAYLOAD_LOCATION, "input-loc1");
        properties.setAttribute(MessagePayloadProxy.SET_PAYLOAD_LOCATION, "output-loc1");
        transformer = new SmooksTransformer(properties);
        transformer.initialise();
        message = MessageFactory.getInstance().getMessage();
        message.getBody().add("input-loc1", stringMessage);
        message = transformer.process(message);
        assertEquals(resultThroughTaskObj, message.getBody().get("output-loc1"));

        return resultThroughTaskObj;
    }



    private ConfigTree createConfig01(String from, String fromType, String to, String toType) {
        ConfigTree properties = new ConfigTree("name");

        properties.setAttribute(SmooksTransformer.RESOURCE_CONFIG, "classpath:/smooks-conf-01.xml");
        // Set the message properties in order to trigger the appropriate transformations
        // on the message...
        properties.setAttribute(SmooksTransformer.FROM, from);
        properties.setAttribute(SmooksTransformer.FROM_TYPE, fromType);
        properties.setAttribute(SmooksTransformer.TO, to);
        properties.setAttribute(SmooksTransformer.TO_TYPE, toType);
        
        return properties;
    }

    private Message createNewMessage(String fromType, String from, String to, String toType) {
        Message message;
        message = MessageFactory.getInstance().getMessage();

        message.getProperties().setProperty(SmooksTransformer.FROM_TYPE, fromType);
        message.getProperties().setProperty(SmooksTransformer.FROM, from);
        message.getProperties().setProperty(SmooksTransformer.TO_TYPE, toType);
        message.getProperties().setProperty(SmooksTransformer.TO, to);

        return message;
    }

    private void assertConfigException(ConfigTree properties, String expectedException) {
        try {
            new SmooksTransformer(properties);
            fail("Expected ConfigurationException: [" + expectedException + "...]");
        } catch (ConfigurationException e) {
            assertTrue("Expected exception message to start with [" + expectedException + "]. Was [" + e.getMessage() + "]", e.getMessage().startsWith(expectedException));
        }
    }
}
