/*
 * JBoss, Home of Professional Open Source Copyright 2006, JBoss Inc., and
 * individual contributors as indicated by the @authors tag. See the
 * copyright.txt in the distribution for a full listing of individual
 * contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.visitors;

import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.milyn.SmooksException;
import org.milyn.cdr.SmooksConfigurationException;
import org.milyn.cdr.SmooksResourceConfiguration;
import org.milyn.container.ExecutionContext;
import org.milyn.delivery.dom.DOMElementVisitor;
import org.w3c.dom.Element;

/**
 * DomServiceDelegatorVisitor is a Smooks vistor which wraps
 * the element object in an ESB Message object and then sends
 * the message to the configured ESB Service.
 * <p>
 * Sample config (smooks-res.xml) :
 * <pre>
 *  &lt;resource-config selector="OrderLines"&gt;
 *  &lt;resource&gt;org.jboss.soa.esb.visitors.DOMServiceDelegateVisitor&lt;/resource&gt;
 *  &lt;param name="service-category-name"&gt;FirstServiceESB&lt;/param&gt;
 *  &lt;param name="service-name"&gt;sendToJMSQueue&lt;/param&gt;
 *  &lt;param name="send-in-visit-after"&gt;true&lt;/param&gt;
 * </pre>
 * 
 * @author <a href="mailto:daniel.bevenius@gmail.com">Daniel Bevenius</a>				
 * @deprecated Use {@link org.jboss.soa.esb.smooks.FragmentRouter}
 */
public class DOMServiceDelegateVisitor implements DOMElementVisitor
{
	public static final String SEND_IN_VISIT_BEFORE = "send-in-visit-before";
	public static final String SEND_IN_VISIT_AFTER = "send-in-visit-after";
	
	private ServiceDelegator serviceDelegator;
	
	private boolean sendInVisitAfter;
	private boolean sendInVisitBefore;
	
	public DOMServiceDelegateVisitor()
	{
		serviceDelegator = new ServiceDelegator();
	}

	public void visitAfter( Element element, ExecutionContext executionContext ) throws SmooksException
	{
		if ( sendInVisitAfter )
			sendToService( element );
	}
	
	public void visitBefore( Element element, ExecutionContext executionContext ) throws SmooksException
	{
		if ( sendInVisitBefore )
			sendToService( element );
	}
	
	protected void sendToService( final Element element ) throws SmooksException
	{
		try
		{
			serviceDelegator.delegateToESBService( createMessage( element ) );
		} 
		catch (MessageDeliverException e)
		{
			String errorMsg = "MessageDeliveryException in visitAfter: ";
			throw new SmooksException( errorMsg, e );
		}
	}
	
	public void setConfiguration( SmooksResourceConfiguration config ) throws SmooksConfigurationException
	{
		serviceDelegator.setConfiguration( config );
		
		sendInVisitAfter = Boolean.parseBoolean( config.getStringParameter( SEND_IN_VISIT_AFTER , "false" ) );
		sendInVisitBefore = Boolean.parseBoolean( config.getStringParameter( SEND_IN_VISIT_BEFORE , "false" ) );
		
		if ( sendInVisitBefore == false && sendInVisitAfter == false )
			throw new SmooksConfigurationException ("Atleast one of [" + SEND_IN_VISIT_BEFORE + "] and [" + SEND_IN_VISIT_AFTER + "] must be true" );
	}

	protected Message createMessage( final Element element )
	{
		Message message = MessageFactory.getInstance().getMessage();
		message.getBody().add( element );
		return message;
	}

	public ServiceDelegator getServiceDelegator()
	{
		return serviceDelegator;
	}

	public void setServiceDelegator( ServiceDelegator serviceDelegator )
	{
		this.serviceDelegator = serviceDelegator;
	}
	
}
