/*
 * JBoss, Home of Professional Open Source
 * Copyright 2008, Red Hat Middleware LLC, and individual contributors
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.soa.esb.services.rules;

import java.util.List;
import java.util.Map;

import org.jboss.internal.soa.esb.services.rules.RuleServiceException;
import org.jboss.soa.esb.helpers.ConfigTree;

import org.jboss.soa.esb.message.Message;

/**
 * RulesEngine Interface. Separates implementation from the runtime, allowing
 * the ESB to support different implementations of rule engines.
 * 
 * @author jdelong@redhat.com
 * 
 */
public interface RuleService {
	
	public void setConfigTree(ConfigTree configTree);

	/**
	 * Execute rules using a certain ruleSet and domain specific language using the Stateless rule engine API
	 * 
	 * @param ruleSet -
	 *            String reference to a file which contains a ruleSet.
	 * @param dsl -
	 *            String reference to a file which contains a custom rule
	 *            language definition
	 * @param ruleReload -
	 *            if set to true, a ruleSet update should result in reloading
	 *            the ruleSet.
	 * @param message -
	 *            Message that is updated with the results.
	 * @param objectList - a list with additional objects (typically pulled from the message) to be inserted into 
	 *            working memory 
	 * 
	 * @return Message with updated objects.
	 * @deprecated Please use {@link #executeStatelessRules(RuleInfo, Message)} instead.
	 */
	public Message executeStatelessRules(
			String ruleSet, 
			String dsl,
			boolean ruleReload, 
			Message message, 
			Map<String,Object> globals, 
			List<Object> objectList) throws RuleServiceException;
	
	public Message executeStatelessRules(final RuleInfo ruleInfo, final Message message) throws RuleServiceException;
	
	/**
	 * Execute rules from a decision table using the Stateless rule engine API
	 * 
	 * @param decisionTable -
	 *            String reference to a file which contains a spreadsheet of rules
	 * @param ruleReload -
	 *            if set to true, a ruleSet update should result in reloading
	 *            the ruleSet.
	 * @param message -
	 *            Message that is updated with the results.
	 * @param objectList - a list with additional objects (typically pulled from the message) to be inserted into 
	 *            working memory 
	 * 
	 * @return Message with updated objects.
	 * @deprecated Please use {@link #executeStatelessRulesFromDecisionTable(RuleInfo, Message)} instead.
	 */
	public Message executeStatelessRulesFromDecisionTable(
			String decisionTable,
			boolean ruleReload, 
			Message message, 
			Map<String,Object> globals, 
			List<Object> objectList) throws RuleServiceException;
	
	public Message executeStatelessRulesFromDecisionTable(final RuleInfo ruleInfo, final Message message) throws RuleServiceException;
	
	/**
	 * Execute rules using a rule package retrieved via the Rule Agent from a URL of local file system
	 *            using the Stateless rule engine API
	 * 
	 * @param ruleAgentProperties -
	 *            String reference to a file which contains properties used by the RuleAgent to locate a rule package.
	 * @param message -
	 *            Message that is updated with the results.
	 * @param objectList - a list with additional objects (typically pulled from the message) to be inserted into 
	 *            working memory 
	 * 
	 * @return Message with updated objects.
	 * @deprecated Please use {@link #executeStatelessRulesFromRuleAgent(RuleInfo, Message)} instead.
	 */
	public Message executeStatelessRulesFromRuleAgent(
			String ruleAgentProperties, 
			Message message, 
			Map<String,Object> globals, 
			List<Object> objectList) throws RuleServiceException;
	
	public Message executeStatelessRulesFromRuleAgent(final RuleInfo ruleInfo, final Message message) throws RuleServiceException;

	/**
	 * Execute rules using a certain ruleSet and domain specific language using the Stateful rule engine API
	 * 
	 * @param ruleSet -
	 *            String reference to a file which contains a ruleSet.
	 * @param dsl -
	 *            String reference to a file which contains a custom rule
	 *            language definition
	 * @param ruleReload -
	 *            if set to true, a ruleSet update should result in reloading
	 *            the ruleSet.
	 * @param dispose -
	 *            if set to true, working memory will be disposed after the rules are fired
	 * @param message -
	 *            Message that is updated with the results.
	 * @param objectList - a list with additional objects (typically pulled from the message) to be inserted into 
	 *            working memory 
	 * 
	 * @return Message with updated objects.
	 * @deprecated Please use {@link #executeStatefulRules(RuleInfo, Message)} instead.
	 */
	public Message executeStatefulRules(
			String ruleSet, 
			String dsl,
			boolean ruleReload, 
			Message message,
			Map<String,Object> globals,
			List<Object> objectList) throws RuleServiceException;
	
	public Message executeStatefulRules(final StatefulRuleInfo info, final Message msg) throws RuleServiceException;
	
	

	/**
	 * Execute rules from a decision table using the Stateful rule engine API
	 * 
	 * @param decisionTable -
	 *            String reference to a file which contains a spreadsheet of rules
	 * @param ruleReload -
	 *            if set to true, a ruleSet update should result in reloading
	 *            the ruleSet.
	 * @param dispose -
	 *            if set to true, working memory will be disposed after the rules are fired
	 * @param message -
	 *            Message that is updated with the results.
	 * @param objectList - a list with additional objects (typically pulled from the message) to be inserted into 
	 *            working memory 
	 * 
	 * @return Message with updated objects.
	 * @deprecated Please use {@link #executeStatefulRulesFromDecisionTable(RuleInfo, Message)} instead.
	 */
	public Message executeStatefulRulesFromDecisionTable(
			String decisionTable,
			boolean ruleReload, 
			Message message, 
			Map<String,Object> globals,
			List<Object> objectList) throws RuleServiceException;
	
	public Message executeStatefulRulesFromDecisionTable(final StatefulRuleInfo info, final Message msg) throws RuleServiceException;

	/**
	 * Execute rules using a rule package retrieved via the Rule Agent from a URL of local file system
	 *            using the Stateful rule engine API
	 * 
	 * @param ruleAgentProperties -
	 *            String reference to a file which contains properties used by the RuleAgent to locate a rule package.
	 * @param dispose -
	 *            if set to true, working memory will be disposed after the rules are fired
	 * @param message -
	 *            Message that is updated with the results.
	 * @param objectList - a list with additional objects (typically pulled from the message)t o be inserted into 
	 *            working memory 
	 * 
	 * @return Message with updated objects.
	 * @deprecated Please use {@link #executeStatefulRulesFromRuleAgent(RuleInfo, Message)} instead.
	 */
	public Message executeStatefulRulesFromRuleAgent(
			String ruleAgentProperties, 
			Message message,
			Map<String,Object> globals,
			List<Object> objectList) throws RuleServiceException;
	
	public Message executeStatefulRulesFromRuleAgent(final StatefulRuleInfo info, final Message msg) throws RuleServiceException;

	/**
	 * Continue executing rules where the working memory already exists.
	 * 
	 * @param rules -
	 *            String reference to a file, either the drl file, the decision table,
	 *            or the ruleAgentProperties. This is used to find the ruleBase.
	 * @param dispose -
	 *            if set to true, working memory will be disposed after the rules are fired
	 * @param message -
	 *            Message that is updated with the results.
	 * @param globals -
	 *            Map of globals variables that should be set in the working memory
	 * @param objectList - a list with additional objects (typically pulled from the message) to be inserted into 
	 *            working memory 
	 * 
	 * @return Message with updated objects.
	 * @deprecated Please use {@link #continueStatefulRulesExecution(StatefulRuleInfo, Message)} instead.
	 */
	public Message continueStatefulRulesExecution(
			String rules,
			boolean dispose, 
			Message message,
			Map<String,Object> globals, 
			List<Object> objectList) throws RuleServiceException;
	
	public Message continueStatefulRulesExecution(final StatefulRuleInfo info, final Message msg) throws RuleServiceException;
	
}
