/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions;

import java.util.List;

import org.jboss.internal.soa.esb.services.rules.RuleInfoBuilder;
import org.jboss.internal.soa.esb.services.rules.RuleServiceCallHelper;
import org.jboss.internal.soa.esb.services.rules.RuleServiceException;
import org.jboss.internal.soa.esb.services.rules.util.RuleConfigUtil;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.mapping.ObjectMappingException;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.routing.MessageRouterException;
import org.jboss.soa.esb.services.rules.RuleInfo;

/**
 * BusinessRulesProcessor extends {@link ContentBasedRouter} but does not perform any routing, it only 
 * executes the business rules.
 * <p/>
 * 
 * Configuration Example:
 *<pre>{@code
 *
 *<action class="org.jboss.soa.esb.actions.BusinessRulesProcessor" name="YourCBRName">
 *    <property name="ruleSet" value="OrderDiscountOnMultipleOrders.drl" />
 *    <property name="ruleReload" value="false" />
 *    <property name="stateful" value="true" />
 *    <property name="object-paths">
 *        <object-path esb="body.TheOrderHeader" />
 *        <object-path esb="body.TheCustomer" />
 *    </property>
 *</action>}
 *
 * As this class extends {@link ContentBasedRouter} please see its javadoc for 
 * configuration descriptions that are common to both classes.
 * 
 * Property description:
 * <lu>
 * <li> <i> class </i> action class, org.jboss.soa.esb.actions.BusinessRulesProcessor
 * <li> <i> stateful </i> Optional property which tells the RuleService to use a stateful session where facts will be remembered between invokations.
 * </lu>
 * </br>
 * 
 * @author John Doe
 * @author <a href="mailto:dbevenius@redhat.com">Daniel Bevenius</a>
 *
 */
public class BusinessRulesProcessor extends ContentBasedRouter
{
    private RuleServiceCallHelper ruleServiceCallHelper;

	public BusinessRulesProcessor(final ConfigTree config) throws ConfigurationException, RegistryException, MessageRouterException
	{
        super(config);
	    ruleServiceCallHelper = new RuleServiceCallHelper(config);
	}
	
	/** 
     * Inspect the content of the message using a rule set 
     *  
     * @param message
     * @return Message
     * @throws ActionProcessingException
	 */
	public Message process(Message message) throws ActionProcessingException
	{
        try 
        {
            final List<Object> objectList = _mapper.createObjectList(message, _messagePathList);
            return executeRulesService( message, objectList );
        } 
        catch (final ObjectMappingException e) 
        {
            throw new ActionProcessingException( e.getMessage(), e);
        } 
        catch (final RuleServiceException e) 
        {
            throw new ActionProcessingException( e.getMessage(), e);
        }
	}
	
	Message executeRulesService(Message message, final List<Object> objectList) throws RuleServiceException 
	{
		final RuleInfoBuilder builder =  ruleServiceCallHelper.getRuleInfoBuilder();
		builder.global("message", message);
		builder.facts(RuleConfigUtil.extractObjectsFromMessage(entryPointMap, message));
		builder.defaultFacts(objectList);
		final RuleInfo ruleInfo = builder.build();
		return ruleServiceCallHelper.executeRulesService(message, ruleInfo);
	}
	
}
