/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.rules.util;

import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.StringReader;

import org.drools.RuleBase;
import org.drools.RuleBaseFactory;
import org.drools.common.DroolsObjectOutputStream;
import org.drools.compiler.DroolsParserException;
import org.drools.compiler.PackageBuilder;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.soa.esb.ConfigurationException;

/**
 * Generates a file containing the a serialized drl in the local filesystem.
 * <p/>
 * This is useful when upgrading Drools version which can make a package
 * compiled with an earlier version invalid.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 *
 */
public class PackageFileGenerator
{
    private PackageFileGenerator()
    {
    }
    
    public static void main(final String... args) throws DroolsParserException, ConfigurationException, IOException
    {
        if (args.length == 0 )
        {
            throw new IllegalArgumentException("Missing arguments to program. " + getUsageString());
        }
        
        final String drlFile = args[0];
        final String pkgFile = args[1];
        
        createPackageFile(drlFile, pkgFile);
    }
    
    public static void createPackageFile(final String drlFile, final String pkgFile) throws DroolsParserException, IOException, ConfigurationException
    {
        AssertArgument.isNotNullAndNotEmpty(drlFile, drlFile);
        AssertArgument.isNotNullAndNotEmpty(pkgFile, pkgFile);
        
        final PackageBuilder builder = new PackageBuilder();
        final String drl = StreamUtils.getResourceAsString(drlFile, "UTF-8");
        builder.addPackageFromDrl(new StringReader(drl));
        
        // gets the package
        org.drools.rule.Package pkg = builder.getPackage();
            
        // creates the rulebase
        RuleBase ruleBase = RuleBaseFactory.newRuleBase();

        // adds the package
        ruleBase.addPackage(pkg);
        
        serializeObject(pkg, pkgFile);
    }
    
    private static void serializeObject(Object object, final String toFile) throws FileNotFoundException, IOException 
    {
        ObjectOutputStream outstream = null;
        try 
        {
            outstream = new DroolsObjectOutputStream(new FileOutputStream(toFile));
            outstream.writeObject(object);
        } 
        finally 
        {
            if (outstream != null) 
            {
                outstream.close();
            }
        }
    }
    
    private static String getUsageString()
    {
        return String.format("Usage: %s <path to drl file> <path to destination package file>", PackageFileGenerator.class.getName());
    }

}
