/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.services.rules;

import java.util.List;
import java.util.Map;

import org.jboss.soa.esb.services.rules.RuleInfo;
import org.jboss.soa.esb.services.rules.RuleService;


/**
 * RuleInfoImpl contains rule execution configuration parameters that 
 * are passed to the {@link RuleService}.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 * @since 4.6
 */
public class RuleInfoImpl implements RuleInfo
{
    /**
	 * String reference to a file which contains rules.
     */
    private final String ruleSource;
    
    /**
     * The type of rules source.
     */
    private final String ruleType;
    
    /**
	 * A String that points to a file containing a Domain Specific Language (DSL).
     */
    private final String dslSource;
    
    /**
	 * If set to true, a ruleSet update should result in reloading the ruleSet.
     */
    private final boolean reload;
    
    /**
	 * Map of globals variables that should be set in the working memory
     */
    private final Map<String, Object> globals;
    
    /**
     * Facts indexed by thier entry point name.
     */
    private final Map<String, List<Object>> factsMap;
    
    /**
     * Facts that are "un-named" and targeted at the default rule working memory.
     */
    private final List<Object> defaultFacts;
    
    /**
     * Sole constructor intended to be called from the {@link Builder}.
     * 
     * @param builder This classes {@link Builder}.
     */
    RuleInfoImpl(final RuleInfoBuilder builder)
    {
        ruleSource = builder.ruleSource;
        ruleType = builder.ruleType;
        dslSource = builder.dslSource;
        globals = builder.globals;
        factsMap = builder.factsMap;
        reload = builder.reload;
        defaultFacts = builder.defaultFacts;
    }

    /* (non-Javadoc)
     * @see org.jboss.soa.esb.services.rules.RuleInfo#getRuleSource()
     */
    public String getRuleSource()
    {
        return ruleSource;
    }
    
    /* (non-Javadoc)
     * @see org.jboss.soa.esb.services.rules.RuleInfo#getRuleType()
     */
    public String getRuleType()
    {
        return ruleType;
    }
    
    /* (non-Javadoc)
     * @see org.jboss.soa.esb.services.rules.RuleInfo#getDslSource()
     */
    public String getDslSource()
    {
        return dslSource;
    }

    /* (non-Javadoc)
     * @see org.jboss.soa.esb.services.rules.RuleInfo#getReload()
     */
    public boolean getReload()
    {
        return reload;
    }
    
    /* (non-Javadoc)
     * @see org.jboss.soa.esb.services.rules.RuleInfo#getGlobals()
     */
    public Map<String, Object> getGlobals()
    {
        return globals;
    }
    
    /* (non-Javadoc)
     * @see org.jboss.soa.esb.services.rules.RuleInfo#getFacts()
     */
    public Map<String, List<Object>> getFacts()
    {
        return factsMap;
    }
    
    public List<Object> getDefaultFacts()
    {
        return defaultFacts;
    }

    public String toString()
    {
        return String.format("%s [ruleSource=%s, dslSource=%s, reload=%b, globals=%s, facts=%s, defaultFacts=%s", getClass().getSimpleName(), ruleSource, dslSource, reload, globals, factsMap, defaultFacts);
    }
}
