/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.jbpm.integration.timer;

import javax.jms.ConnectionFactory;
import javax.jms.Destination;
import javax.jms.JMSException;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import org.jbpm.JbpmConfiguration;
import org.jbpm.JbpmException;
import org.jbpm.svc.Service;
import org.jbpm.svc.ServiceFactory;

/**
 * Implementation of a scheduler service using quartz.
 */
public class JmsSchedulerServiceFactory implements ServiceFactory
{
    /**
     * Serial Version UID for this class.
     */
    private static final long serialVersionUID = 1642915732295620551L;
    
    /**
     * The JNDI name for locating the JMS connection factory.
     */
    String connectionFactoryJndiName = "java:/JmsXA";
    /**
     * The JNDI name for locating the jBPM timer queue destination.
     */
    String destinationJndiName = "queue/JbpmTimerQueue";
    /**
     * A flag indicating whether commit should be invoked on the JMS session.
     */
    boolean isCommitEnabled = false;

    /**
     * The JMS connection factory associated with the scheduler.
     */
    private ConnectionFactory connectionFactory;
    /**
     * The JMS destination for the scheduler.
     */
    private Destination destination;

    /**
     * Get the JMS connection factory associated with this scheduler.
     * @return The associated JMS connection factory.
     */
    public ConnectionFactory getConnectionFactory()
    {
      if (connectionFactory == null)
      {
        try
        {
          connectionFactory = (ConnectionFactory)lookup(connectionFactoryJndiName);
        }
        catch (NamingException e)
        {
          throw new JbpmException("could not retrieve message connection factory", e);
        }
      }
      return connectionFactory;
    }

    /**
     * Get the JMS destination associated with this scheduler.
     * @return The associated JMS destination.
     */
    public Destination getDestination()
    {
      if (destination == null)
      {
        try
        {
          destination = (Destination)lookup(destinationJndiName);
        }
        catch (NamingException e)
        {
          throw new JbpmException("could not retrieve job destination", e);
        }
      }
      return destination;
    }

    /**
     * Get the commitEnabled flag.
     * @return true if commit is to be invoked on the JMS session, false otherwise.
     */
    public boolean isCommitEnabled()
    {
      return isCommitEnabled;
    }

    /**
     * Lookup a name within JNDI.
     * @param name The JNDI name to locate.
     * @return The object bound to the name.
     * @throws NamingException For errors locating the specified object within JNDI.
     */
    private static Object lookup(String name) throws NamingException
    {
      Context initial = new InitialContext();
      try
      {
        return initial.lookup(name);
      }
      finally
      {
        initial.close();
      }
    }

    /**
     * Create a JMS scheduler service based on this factory.
     */
    public Service openService()
    {
      try
      {
        return new JmsSchedulerService(JbpmConfiguration.getInstance().getCurrentJbpmContext(),
            getConnectionFactory(), getDestination(), isCommitEnabled);
      }
      catch (JMSException e)
      {
        throw new JbpmException("couldn't open message session", e);
      }
    }

    /**
     * Close this factory.
     */
    public void close()
    {
      connectionFactory = null;
      destination = null;
    }
}
