/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.jbpm.integration.job;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.jboss.soa.esb.services.jbpm.integration.msg.RetryExecutor;
import org.jbpm.JbpmContext;
import org.jbpm.command.Command;
import org.jbpm.graph.exe.ProcessInstance;
import org.jbpm.job.Job;

/**
 * Individual job processing command.
 * @author Alejandro Guizar
 */
public class ExecuteJobCommand implements Command {

  /**
   * The job id.
   */
  private final long jobId;
  /**
   * The redelivered flag.
   */
  private final boolean redelivered ;

  private static final long serialVersionUID = 1L;

  /**
   * Create the job command with a specified job id.
   * @param jobId The associated job id.
   * @param redelivered The redelivered flag.
   */
  public ExecuteJobCommand(long jobId, final boolean redelivered) {
    this.jobId = jobId;
    this.redelivered = redelivered ;
  }

  /**
   * Execute the command.
   * @param jbpmContext The jBPM context associated with the execution.
   * @throws Exception for errors during execution.
   */
  public Object execute(JbpmContext jbpmContext) throws Exception {
    Job job = acquireJob(jbpmContext);
    if (JobUtil.isDeleted(job)) {
      if (log.isDebugEnabled()) {
        log.debug("job " + jobId + " was deleted");
      }
      return null;
    }
    if (job.isSuspended()) {
      if (log.isDebugEnabled()) {
        log.debug(job + " is suspended");
      }
      RetryExecutor.handleSuspendedJob(job) ;
      return null ;
    }
    if (redelivered) {
        if (job.getRetries() > 0) {
            job.setRetries(job.getRetries() - 1);
            if (log.isDebugEnabled()) {
              log.debug("Rescheduling job " + job.getId());
            }
            job.setLockOwner(null) ;
            jbpmContext.getServices().getMessageService().send(job) ;
        } else {
            log.error("Job retry count exceeded for job " + job.getId());
        }
        return null ;
    }
    executeJob(job, jbpmContext);
    return job;
  }

  private Job acquireJob(JbpmContext jbpmContext) {
    Job job = jbpmContext.getJobSession().loadJob(jobId);

    // register process instance for automatic save
    // see https://jira.jboss.org/jira/browse/JBPM-1015
    ProcessInstance processInstance = job.getProcessInstance();
    jbpmContext.addAutoSaveProcessInstance(processInstance);

    // if job is exclusive, lock process instance
    if (job.isExclusive()) {
        jbpmContext.getGraphSession().lockProcessInstance(processInstance);
    }

    // mark job as locked to prevent other parts of the engine from deleting it
    job.setLockOwner(toString());
    return job;
  }

  /**
   * Execute the specific job.
   * @param job The job to execute.
   * @param jbpmContext The jBPM context associated with the execution.
   */
  static void executeJob(Job job, JbpmContext jbpmContext) {
    if (log.isDebugEnabled()) {
      log.debug("executing " + job);
    }
    try {
      if (job.execute(jbpmContext)) {
        jbpmContext.getJobSession().deleteJob(job);
      }
    }
    catch (Exception e) {
      if (log.isDebugEnabled()) {
        log.debug("exception while executing " + job, e);
      }
      jbpmContext.setRollbackOnly();
    }
  }

  /**
   * The log for this class.
   */
  private static Log log = LogFactory.getLog(ExecuteJobCommand.class);
}
