/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.testutils;

import com.arjuna.common.util.propertyservice.PropertyManager;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.JAXRRegistryImpl;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.common.ModulePropertyManager;
import org.jboss.soa.esb.dom.YADOMUtil;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.config.ConfigurationController;
import org.jboss.soa.esb.listeners.config.Generator;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleController;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.parameters.ParamRepositoryException;
import org.jboss.soa.esb.services.registry.Registry;
import org.jboss.soa.esb.services.registry.RegistryException;
import org.jboss.soa.esb.services.registry.RegistryFactory;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.sql.SQLException;
import java.util.Hashtable;

/**
 * Utility class for working with the ESB configuration.
 *
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class ESBConfigUtil {

    private Generator generator;
    private Document listenerConfig;
    private Document gatewayConfig;
    private ManagedLifecycleController controller;
    private boolean installMockCourierFactory = true;
    private boolean installMockRegistry = true;
    private Hashtable<String, PropertyManager> originalPropertyManagers;
    private Registry originalRegistry;

    public ESBConfigUtil(InputStream esbXsdConfig) throws IOException, ConfigurationException, SAXException {
        AssertArgument.isNotNull(esbXsdConfig, "esbXsdConfig");
        ByteArrayOutputStream listenerConfigStream = new ByteArrayOutputStream();
        ByteArrayOutputStream gatewayConfigStream = new ByteArrayOutputStream();
        generator = null;

        generator = new Generator(esbXsdConfig, listenerConfigStream, gatewayConfigStream);
        generator.generate();
        listenerConfig = YADOMUtil.parseStream(new ByteArrayInputStream(listenerConfigStream.toByteArray()), false, false);
        gatewayConfig = YADOMUtil.parseStream(new ByteArrayInputStream(gatewayConfigStream.toByteArray()), false, false);
    }

    public void setESBProperties(InputStream esbProperties) throws IOException, ConfigurationException, SAXException {
        originalPropertyManagers = (Hashtable<String, PropertyManager>) ModulePropertyManager.getManagers().clone();
        ModulePropertyManager.configure(esbProperties);
    }

    public void resetESBProperties() {
        if(originalPropertyManagers != null) {
            // The ModulePropertyManager uses statics, so other tests (to
            // follow) may be unwittingly dependent on its state. We reset
            // it back to what it was here...
            ModulePropertyManager.setManagers(originalPropertyManagers);
            originalPropertyManagers = null;
        }
    }

    public void installRegistry() {
        originalRegistry = RegistryFactory.getRegistrySingleton();
        Registry newRegistry = null;
        try {
            newRegistry = RegistryFactory.createRegistry();
            RegistryFactory.setRegistry(newRegistry);
        } catch (RegistryException e) {
            throw new RuntimeException("Failed to create registry instance.", e);
        }
        
        if(newRegistry instanceof JAXRRegistryImpl) {
            try {
                TestEnvironmentUtil.startJAXRDatabase();
            } catch (SQLException e) {
                throw new RuntimeException("Failed to start JAXR Database.", e);
            }
            installMockCourierFactory = false;
            installMockRegistry = false;
        }
    }

    public void uninstallRegistry() {
        try {
            Registry currentRegistry = RegistryFactory.getRegistrySingleton();
            if(currentRegistry instanceof JAXRRegistryImpl) {
                try {
                    TestEnvironmentUtil.stopJAXRDatabase();
                } catch (Exception e) {
                    throw new RuntimeException("Failed to stop JAXR Database.", e);
                }
            }
        } finally {
            RegistryFactory.setRegistry(originalRegistry);
        }
    }

    public boolean isInstallMockCourierFactory() {
        return installMockCourierFactory;
    }

    public void setInstallMockCourierFactory(boolean installMockCourierFactory) {
        this.installMockCourierFactory = installMockCourierFactory;
    }

    public boolean isInstallMockRegistry() {
        return installMockRegistry;
    }

    public void setInstallMockRegistry(boolean installMockRegistry) {
        this.installMockRegistry = installMockRegistry;
    }

    public ConfigTree getListenerConfig(String listenerName) {
        AssertArgument.isNotNull(listenerName, "listenerName");
        return ConfigTree.fromElement(getListenerConfig(listenerName, listenerConfig));
    }

    public ConfigTree getGatewayConfig(String gatewayName) {
        AssertArgument.isNotNull(gatewayName, "gatewayName");
        return ConfigTree.fromElement(getListenerConfig(gatewayName, gatewayConfig));
    }

    private Element getListenerConfig(String name, Document config) {
        NodeList listenerConfigElements = config.getDocumentElement().getElementsByTagName(name);

        if(listenerConfigElements == null || listenerConfigElements.getLength() == 0) {
            listenerConfigElements = config.getDocumentElement().getElementsByTagName("listener");
            if(listenerConfigElements == null || listenerConfigElements.getLength() == 0) {
                return null;
            }

            for(int i = 0; i < listenerConfigElements.getLength(); i++) {
                Element element = (Element) listenerConfigElements.item(i);
                if(name.equals(element.getAttribute("name"))) {
                    return element;
                }
            }

            return null;
        }

        return (Element) listenerConfigElements.item(0);
    }

    public ConfigTree getActionConfig(String listenerName, String actionName) {
        AssertArgument.isNotNull(listenerName, "listenerName");
        AssertArgument.isNotNull(actionName, "actionName");

        Element listener = getListenerConfig(listenerName, listenerConfig);
        NodeList actions = listener.getElementsByTagName("action");

        if(actions == null || actions.getLength() == 0) {
            return null;
        }

        for(int i = 0; i < actions.getLength(); i++) {
            Element action = (Element) actions.item(i);

            if(actionName.equals(action.getAttribute("action"))) {
                return ConfigTree.fromElement(action);
            }
        }
        
        return null;
    }

    public Document getListenerConfig() {
        return listenerConfig;
    }

    public Document getGatewayConfig() {
        return gatewayConfig;
    }

    public void startController() throws ParamRepositoryException, ConfigurationException, ManagedLifecycleException, SAXException {
        if(controller != null) {
            throw new RuntimeException("Sorry, this ESBConfigUtil instance has already been started/used.  You cannot restart!");
        }

        ConfigTree listenerConfigTree = ConfigTree.fromElement(listenerConfig.getDocumentElement());
        ConfigTree gatewayConfigTree = ConfigTree.fromElement(gatewayConfig.getDocumentElement());

        if(installMockCourierFactory) {
            MockCourierFactory.install();
        }
        if(installMockRegistry) {
            MockRegistry.install();
        }

        controller = ConfigurationController.startController(generator.getModel(), listenerConfigTree, gatewayConfigTree);
    }

    public void stopController() {
        try {
            ConfigurationController.stopController(controller);
        } finally {
            try {
                try {
                    if(installMockRegistry) {
                        MockRegistry.uninstall();
                    }
                } finally {
                    if(installMockCourierFactory) {
                        MockCourierFactory.uninstall();
                    }
                }
            } finally {
                resetESBProperties();
            }
        }
    }
}
