/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.testutils;

import junit.framework.TestCase;
import org.jboss.internal.soa.esb.assertion.AssertArgument;
import org.jboss.soa.esb.common.ModulePropertyManager;

import java.io.InputStream;

/**
 * Abstract Test Runner class.
 * 
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public abstract class AbstractTestRunner {

    private InputStream serviceConfig;
    private InputStream esbProperties;

    public AbstractTestRunner setServiceConfig(InputStream serviceConfig) {
        AssertArgument.isNotNull(serviceConfig, "serviceConfig");
        this.serviceConfig = serviceConfig;
        return this;
    }

    public AbstractTestRunner setServiceConfig(String serviceConfig) {
        AssertArgument.isNotNull(serviceConfig, "serviceConfig");
        setServiceConfig(getClass().getResourceAsStream(serviceConfig));
        return this;
    }

    public AbstractTestRunner setEsbProperties(InputStream esbProperties) {
        AssertArgument.isNotNull(esbProperties, "esbProperties");
        this.esbProperties = esbProperties;
        return this;
    }

    public AbstractTestRunner setEsbProperties(String esbProperties) {
        AssertArgument.isNotNull(esbProperties, "esbProperties");
        setEsbProperties(getClass().getResourceAsStream(esbProperties));
        return this;
    }

    public abstract void test() throws Exception;

    public void run() throws Exception {
        if(serviceConfig == null) {
            TestCase.fail("'serviceConfig' not configured.  Must call setServiceConfig().");
        }

        ESBConfigUtil esbConfig = new ESBConfigUtil(serviceConfig);

        if(esbProperties != null) {
            esbConfig.setESBProperties(esbProperties);
        } else {
            esbConfig.setESBProperties(AbstractTestRunner.class.getResourceAsStream("jbossesb-properties.xml"));
        }

        String installedJaxrConnectionFactoryImpl = System.getProperty( "javax.xml.registry.ConnectionFactoryClass");
        String jaxrConnectionFactoryImpl = ModulePropertyManager.getPropertyManager("registry").getProperty("org.jboss.soa.esb.registry.factoryClass", "org.apache.ws.scout.registry.ConnectionFactoryImpl"); 
        System.setProperty( "javax.xml.registry.ConnectionFactoryClass", jaxrConnectionFactoryImpl);
        try {
            esbConfig.installRegistry();
            esbConfig.startController();
            try {
                test();
            } finally {
                esbConfig.stopController();
            }
        } finally {
            esbConfig.uninstallRegistry();
            if(installedJaxrConnectionFactoryImpl != null) {
                System.setProperty("javax.xml.registry.ConnectionFactoryClass", installedJaxrConnectionFactoryImpl);
            } else {
                System.getProperties().remove("javax.xml.registry.ConnectionFactoryClass");
            }
            esbConfig.resetESBProperties();
        }
    }
}
