/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.services.security.auth.ws;

import static org.junit.Assert.assertTrue;

import java.security.PrivilegedAction;

import javax.security.auth.Subject;

import junit.framework.JUnit4TestAdapter;

import org.picketlink.identity.federation.core.wstrust.SamlCredential;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.junit.Test;
import org.milyn.FilterSettings;
import org.milyn.Smooks;
import org.milyn.StreamFilterType;
import org.milyn.payload.StringResult;
import org.milyn.payload.StringSource;

/**
 * Unit test for {@link SamlVisitor}
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 */
public class SamlVisitorUnitTest
{
    private static final String SECURITY_NS = "http://security";
    private static final String HEADER_NS = "http://header";
    private static final String SECURITY_START = "<t:Security xmlns:t=\"" + SECURITY_NS + "\">";
    private static final String HEADER_START = "<h:Header xmlns:h=\"" + HEADER_NS + "\">";

    @Test
    public void addAssertionToEmptySecurityHeader() throws Exception
    {
        final Smooks smooks = createSmooks("Envelope/Header/Security");

        final StringSource source = new StringSource("<Envelope><Header>" + SECURITY_START + "</t:Security></Header></Envelope>");
        final StringResult result = new StringResult();

        final SamlCredential samlPrincipal = new SamlCredential("<dummyAssertion/>");
        final Subject subject = new Subject();
        subject.getPublicCredentials().add(samlPrincipal);
        
        final String expected = "<Envelope><Header>" + SECURITY_START + "<dummyAssertion/></t:Security></Header></Envelope>";
        final String actual = (String) Subject.doAs(subject, new PrivilegedAction<String>()
        {
            public String run()
            {
		        smooks.filterSource(source, result);
                return result.toString();
            }
        });
        
        assertTrue(XMLHelper.compareXMLContent(expected, actual));
    }

    /**
     * This test shows that it's is possible have the the {@link SamlVisitor#getHeaderToInsert()
     * method return null which will simply produce an unchanged security header.
     */
    @Test
    public void addNoAssertionToEmptySecurityHeader() throws Exception
    {
        final Smooks smooks = createSmooks("Envelope/Header/Security");

        final String expected = "<Envelope><Header>" + SECURITY_START + "</t:Security></Header></Envelope>";
        final StringSource source = new StringSource(expected);
        final StringResult result = new StringResult();

        smooks.filterSource(source, result);

        final String actual = result.toString();

        assertTrue(XMLHelper.compareXMLContent(expected, actual));
    }

    @Test
    public void addAssertionToSOAPHeader() throws Exception
    {
        final Smooks smooks = createSmooks("Envelope/Header");

        final String soap = "<Envelope>" + HEADER_START + "</h:Header></Envelope>";
        final StringSource source = new StringSource(soap);
        final StringResult result = new StringResult();
        
        final SamlCredential samlPrincipal = new SamlCredential("<dummyAssertion/>");
        final Subject subject = new Subject();
        subject.getPublicCredentials().add(samlPrincipal);
        
        final String expected = "<Envelope>" + HEADER_START + SECURITY_START + "<dummyAssertion/></t:Security></h:Header></Envelope>";
        final String actual = (String) Subject.doAs(subject, new PrivilegedAction<String>()
        {
            public String run()
            {
                smooks.filterSource(source, result);
                return result.toString();
            }
        });

        assertTrue(XMLHelper.compareXMLContent(expected, actual));
    }
    
    @Test
    public void configFromFileWithSecurityHeader() throws Exception
    {
        final Smooks smooks = new Smooks(getClass().getResourceAsStream("smooks-saml-injector.xml"));
        
        final String soap = StreamUtils.readStreamString(getClass().getResourceAsStream("saml-inject.xml"), "UTF-8");
        final StringSource source = new StringSource(soap);
        final StringResult result = new StringResult();
        
        final SamlCredential samlPrincipal = new SamlCredential("<dummyAssertion/>");
        final Subject subject = new Subject();
        subject.getPublicCredentials().add(samlPrincipal);

        final String expected = StreamUtils.readStreamString(getClass().getResourceAsStream("saml-inject-expected.xml"), "UTF-8");
        final String actual = (String) Subject.doAs(subject, new PrivilegedAction<String>()
        {
            public String run()
            {
                smooks.filterSource(source, result);
                return result.toString();
            }
        });
        
        assertTrue(XMLHelper.compareXMLContent(expected, actual));
    }
    
    @Test
    public void configFromFileWithoutSecurityHeader() throws Exception
    {
        final Smooks smooks = new Smooks(getClass().getResourceAsStream("smooks-saml-injector.xml"));
        
        final String soap = StreamUtils.readStreamString(getClass().getResourceAsStream("saml-inject-no-security-header.xml"), "UTF-8");
        final StringSource source = new StringSource(soap);
        final StringResult result = new StringResult();
        
        final SamlCredential samlPrincipal = new SamlCredential("<dummyAssertion/>");
        final Subject subject = new Subject();
        subject.getPublicCredentials().add(samlPrincipal);

        final String expected = StreamUtils.readStreamString(getClass().getResourceAsStream("saml-inject-expected.xml"), "UTF-8");
        final String actual = (String) Subject.doAs(subject, new PrivilegedAction<String>()
        {
            public String run()
            {
                smooks.filterSource(source, result);
                return result.toString();
            }
        });
        
        assertTrue(XMLHelper.compareXMLContent(expected, actual));
    }

    private Smooks createSmooks(final String selector)
    {
        final Smooks smooks = new Smooks();
        smooks.setFilterSettings(new FilterSettings().setFilterType(StreamFilterType.SAX));

        final SOAPSecurityHeaderVisitor samlVisitor = new SamlVisitor();
        samlVisitor.setSecurityElementName("Security").setSecurityNS(SECURITY_NS).setSecurityPrefix("t").setAddSecurityXmlNSAttribute(true);
        samlVisitor.setHeaderNS(HEADER_NS).setHeaderElementName("Header");

        smooks.addVisitor(samlVisitor, selector);
        return smooks;
    }

    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(SamlVisitorUnitTest.class);
    }
}
