/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.message;

import static org.junit.Assert.*;
import junit.framework.JUnit4TestAdapter;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.junit.Test;

/**
 * Test for {@link BeanConfigActionProcessor}.
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 *
 */
public class BeanConfigActionProcessorUnitTest
{
    @Test (expected = IllegalArgumentException.class)
    public void constructorNullConfig() throws ConfigurationException
    {
        new BeanConfigActionProcessor(null, getClass());
    }
    
    @Test (expected = IllegalArgumentException.class)
    public void constructorNullActionClass() throws ConfigurationException
    {
        new BeanConfigActionProcessor(new ConfigTree("test"), null);
    }
    
    @Test
    public void processActionClass() throws ActionProcessingException, ConfigurationException
    {
        Class<?> actionClass = TestAction2.class;
        ConfigTree actionConfig = createActionConfig(actionClass.getName());
        Message processedMsg = process(actionConfig, actionClass);
        
        TestAction2 testAction = (TestAction2) processedMsg.getBody().get();
        assertEquals("Daniel", testAction.getFirstName());
    }
    
    @Test
    public void processActionClassWithServiceInformation() throws ActionProcessingException, ConfigurationException
    {
        Class<?> actionClass = TestAction.class;
        ConfigTree actionConfig = createConfig(actionClass.getName());
        Message processedMsg = process(actionConfig, actionClass);
        
        TestAction testAction = (TestAction) processedMsg.getBody().get();
        assertEquals("Daniel", testAction.getFirstName());
        assertEquals("TestCategory", testAction.getServiceCategory());
        assertEquals("TestService", testAction.getServiceName());
    }
    
    private Message process(final ConfigTree config, final Class<?> actionClass) throws ConfigurationException, ActionProcessingException
    {
        BeanConfigActionProcessor processor = new BeanConfigActionProcessor(config, actionClass);
        Message message = MessageFactory.getInstance().getMessage();
        return processor.process(message);
    }
    
    private ConfigTree createConfig(final String actionClass)
    {
        ConfigTree serviceConfig = new ConfigTree("listener");
        serviceConfig.setAttribute("listener-class", "org.jboss.soa.esb.listeners.message.MessageAwareListener");
        serviceConfig.setAttribute("max-threads", "1");
        serviceConfig.setAttribute("mep", "OneWay");
        serviceConfig.setAttribute("name", "JMS-ESBListener");
        serviceConfig.setAttribute("service-category", "TestCategory");
        serviceConfig.setAttribute("service-name", "TestService");
        serviceConfig.setAttribute("service-description", "Some description of the service");
        return createActionConfig(actionClass, serviceConfig);
    }
    
    private ConfigTree createActionConfig(final String actionClass)
    {
        return createActionConfig(actionClass, null);
    }
    
    private ConfigTree createActionConfig(final String actionClass, final ConfigTree parent)
    {
        ConfigTree actionConfig = parent == null ? new ConfigTree("action") : new ConfigTree("action", parent);
        actionConfig.setAttribute("action", "testAction");
        actionConfig.setAttribute("class", actionClass);
        actionConfig.setAttribute("firstName", "Daniel");
        return actionConfig;
    }
    
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(BeanConfigActionProcessorUnitTest.class);
    }

}
