/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2006, JBoss Inc.
 */
package org.jboss.soa.esb.listeners.gateway;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.net.InetAddress;
import java.net.URI;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.couriers.MockCourier;
import org.jboss.internal.soa.esb.couriers.MockCourierFactory;
import org.jboss.internal.soa.esb.services.registry.MockRegistry;
import org.jboss.remoting.Client;
import org.jboss.remoting.InvokerLocator;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.addressing.EPR;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.listeners.ListenerTagNames;
import org.jboss.soa.esb.listeners.lifecycle.ManagedLifecycleException;
import org.jboss.soa.esb.listeners.message.MessageDeliverException;
import org.jboss.soa.esb.listeners.message.UncomposedMessageDeliveryAdapter;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.xml.sax.SAXException;

/**
 * JBossRemotingGatewayListenerUnit unit tests.
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class JBossRemotingGatewayListenerUnitTest {
	private Logger log = Logger
			.getLogger( JBossRemotingGatewayListenerUnitTest.class );

    private JBossRemotingGatewayListener listener;

    private MockCourier courier1;
    private MockCourier courier2;
    private EPR epr3 = new EPR();

    @Before
    public void setUp() throws Exception {
        MockCourierFactory.install();
        MockRegistry.install();
        
        EPR epr2 = new EPR(new URI("2"));
        courier1 = new MockCourier(false); // Will fail
        courier2 = new MockCourier(true);  // Will work
        MockRegistry.register("cat", "servicex", courier1);
        MockRegistry.register("cat", "servicex", epr2, courier2);
     }
    @After
    public void tearDown() throws Exception {
        MockRegistry.uninstall();
        MockCourierFactory.uninstall();
    }
    @Test
    public void test_config() throws SAXException {
        test_config(null, "servicex", "http", "8888", "");
        test_config("",   "servicex", "http", "8888", "");
        test_config("cat", null,      "http", "8888", "");
        test_config("cat", "",        "http", "8888", "");
        test_config("cat", "servicex", null,  "8888", "");
        test_config("cat", "servicex", "",    "8888", "");
        test_config("cat", "servicex", "http", null,  "");
        test_config("cat", "servicex", "http", "",    "");
    }
    @Test
    public void test_lifecycle() throws SAXException, ConfigurationException, ManagedLifecycleException {
        ConfigTree config = getConfig("cat", "servicex", "http", "8887", -1L);

        listener = new JBossRemotingGatewayListener(config);

        // try starting without initialising...
        try {
            listener.start();
            fail("Expected ManagedLifecycleException");
        } catch(ManagedLifecycleException e) {}

        // Initialise...
        listener.initialise();

        // try initialising again...
        listener.initialise();

        // try stopping without starting...
        try {
            listener.stop();
            fail("Expected ManagedLifecycleException");
        } catch(ManagedLifecycleException e) {}

        // Start...
        listener.start();

        // try starting again...
        listener.start();

        // Stop...
        listener.stop();

        // try stopping again...
        listener.stop();
    }
    @Test
    public void test_http_async() throws Throwable {
        test_delivery("http", false, "8888");
    }

//    public void x_test_http_sync() throws Throwable {
//        test_delivery("http", true);
//        System.out.println("Ho " + courier3);
//    }
    @Test
    public void test_socket_() throws Throwable {
        test_delivery("socket", false, "8886");
    }
    
    public void test_delivery(String protocol, boolean synchronous, final String port) throws Throwable {
        startServer(protocol, synchronous, port);
        sendMessageToServer(protocol, protocol + "_payload", port);
        stopServer();
    }

    @Test
    public void test_binding_no_host() throws Exception {
        final String origBinding = System.getProperty(Environment.ESB_BIND_ADDRESS) ;
        final String host = "binding.example.com" ;
        final String protocol = "http" ;
        final String port = "8887" ;
        final URI expected = new URI(protocol + "://" + host + ":" + port) ;
        System.setProperty(Environment.ESB_BIND_ADDRESS, host) ;
        try {
            final ConfigTree config = getConfig("cat", "servicex", "http", "8887", -1L);
            
            final JBossRemotingGatewayListener listener = new JBossRemotingGatewayListener(config);
            final URI locatorURI = listener.getJbrServerLocatorURI() ;
            assertEquals("Locator URI", expected, locatorURI) ;
        } finally {
            if (origBinding == null) {
                System.clearProperty(Environment.ESB_BIND_ADDRESS) ;
            } else {
                System.setProperty(Environment.ESB_BIND_ADDRESS, origBinding) ;
            }
        }
    }

    @Test
    public void test_binding_empty_host() throws Exception {
        final String origBinding = System.getProperty(Environment.ESB_BIND_ADDRESS) ;
        final String host = "binding.example.com" ;
        final String protocol = "http" ;
        final String port = "8887" ;
        final URI expected = new URI(protocol + "://" + host + ":" + port) ;
        System.setProperty(Environment.ESB_BIND_ADDRESS, host) ;
        try {
            final ConfigTree config = getConfig("cat", "servicex", "http", "8887", -1L);
            config.setAttribute(JBossRemotingGatewayListener.JBR_SERVER_HOST, "");
            
            final JBossRemotingGatewayListener listener = new JBossRemotingGatewayListener(config);
            final URI locatorURI = listener.getJbrServerLocatorURI() ;
            assertEquals("Locator URI", expected, locatorURI) ;
        } finally {
            if (origBinding == null) {
                System.clearProperty(Environment.ESB_BIND_ADDRESS) ;
            } else {
                System.setProperty(Environment.ESB_BIND_ADDRESS, origBinding) ;
            }
        }
    }

    public void x_test_profile() throws Throwable {
        startServer("http", false, "8889");
        for(int i = 0; i < 10000; i++) {
            sendMessageToServer("http", "xxx_payload", "8889");
            MockCourierFactory.resetCouriers();
            Thread.sleep(50);
        }
        stopServer();
    }

    @SuppressWarnings("serial")
    private void startServer(String protocol, boolean synchronous, final String port) throws SAXException, ConfigurationException, ManagedLifecycleException {
        ConfigTree config = getConfig("cat", "servicex", protocol, port, -1L);

        if(synchronous) {
            listener = new JBossRemotingGatewayListener(config) {
                protected UncomposedMessageDeliveryAdapter createDeliveryAdapter() throws ConfigurationException {
                    try {
                        return new UncomposedMessageDeliveryAdapter("cat", "servicex", new JBossRemotingMessageComposer()) {
                            @SuppressWarnings("unused")
                            protected EPR getReplyToAddress(EPR toEpr) throws ConfigurationException {
                                return epr3;
                            }
                        };
                    } catch (MessageDeliverException e) {
                        e.printStackTrace();
                        fail(e.getMessage());
                        return null;
                    }
                }
            };
        } else {
            config.setAttribute("synchronous", "false");
            listener = new JBossRemotingGatewayListener(config);
        }

        listener.initialise();
        listener.start();
    }

    private void sendMessageToServer(String protocol, Object messagePayload, final String port) throws Throwable {
       
        String locatorURI = protocol + "://" + InetAddress.getLocalHost().getHostName() + ":" + port;
        InvokerLocator locator = new InvokerLocator(locatorURI);
        System.out.println("Calling remoting server with locator uri of: " + locatorURI);

        Client remotingClient = null;
        try {
            remotingClient = new Client(locator);
            remotingClient.connect();

            // Make sure the courier doesn't have a payload beforehand...
            assertEquals(null, courier2.message);
            
            // Deliver the message to the listener...
            Object response = remotingClient.invoke(messagePayload);
            assertEquals("<ack/>", response);

            // Make sure the courier payload after matches the input...
            Object courierPayloadAfter = courier2.message.getBody().get();
            assertEquals(messagePayload, courierPayloadAfter);
        } finally {
            if(remotingClient != null) {
                remotingClient.disconnect();
            }
        }
    }

    private void stopServer() throws ManagedLifecycleException {
        if(listener != null && listener.isStarted()) {
            try {
                listener.stop();
            } finally {
                listener.destroy();
            }
        }
    }

    private ConfigTree getConfig(String category, String service, String protocol, String port, long timeout) throws SAXException {
        ConfigTree config = ConfigTree.fromXml("<listener/>");

        config.setAttribute(ListenerTagNames.TARGET_SERVICE_CATEGORY_TAG, category);
        config.setAttribute(ListenerTagNames.TARGET_SERVICE_NAME_TAG, service);
        config.setAttribute(JBossRemotingGatewayListener.JBR_SERVER_PROTOCOL, protocol);
        config.setAttribute(JBossRemotingGatewayListener.JBR_SERVER_PORT, port);
        config.setAttribute(ListenerTagNames.SERVICE_INVOKER_TIMEOUT, String.valueOf(timeout));
        
        return config;
    }

    private void test_config(String category, String service, String protocol, String port, String exception) throws SAXException {
        ConfigTree config = getConfig(category, service, protocol, port, -1L);

        try {
            listener = new JBossRemotingGatewayListener(config);
            listener.initialise();
            fail("Expected a ManagedLifecycleException.");
        } catch (ConfigurationException e) {
        	log.error(e);
            fail("Unexpected ConfigurationException. " + e.getMessage());
        } catch (ManagedLifecycleException e) {
            Throwable cause = e.getCause();

            assertTrue(cause instanceof ConfigurationException);
            assertTrue(cause.getMessage().startsWith(exception));
        }
    }
    
    public static junit.framework.Test suite() {
        return new JUnit4TestAdapter(JBossRemotingGatewayListenerUnitTest.class);
    }
}
