/*
 * JBoss, Home of Professional Open Source Copyright 2009, Red Hat Middleware
 * LLC, and individual contributors by the @authors tag. See the copyright.txt
 * in the distribution for a full listing of individual contributors.
 * 
 * This is free software; you can redistribute it and/or modify it under the
 * terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 * 
 * This software is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License for more
 * details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this software; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA, or see the FSF
 * site: http://www.fsf.org.
 */
package org.jboss.soa.esb.listeners.deployers.mc;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

import org.jboss.deployers.client.spi.Deployment;
import org.jboss.deployers.plugins.main.MainDeployerImpl;
import org.jboss.deployers.spi.DeploymentException;
import org.jboss.deployers.structure.spi.DeploymentUnit;
import org.jboss.deployers.vfs.spi.client.VFSDeploymentFactory;
import org.jboss.deployers.vfs.spi.structure.VFSDeploymentUnit;
import org.jboss.metadata.web.jboss.JBossServletsMetaData;
import org.jboss.metadata.web.jboss.JBossWebMetaData;
import org.jboss.metadata.web.spec.ServletMappingMetaData;
import org.jboss.soa.esb.listeners.deployers.mc.util.VfsUtil;
import org.jboss.test.kernel.junit.MicrocontainerTest;
import org.jboss.virtual.VFS;
import org.jboss.virtual.VirtualFile;

/**
 * Unit tests for {@link EsbWebServiceDeployer}.
 * <p/>
 * 
 * @author <a href="mailto:dbevenius@jboss.com">Daniel Bevenius</a>
 *
 */
public class EsbWebServiceDeployerUnitTest extends MicrocontainerTest
{
    private Deployment deployment;
    private MainDeployerImpl mainDeployer;
    
    public EsbWebServiceDeployerUnitTest(String name)
    {
        super(name);
    }
    
    @Override
    public void afterSetUp() throws Exception
    {
        super.afterSetUp();
        
        final String archiveName = "exploded-esb-archive.esb";
        final URL dir = getClass().getResource("/org/jboss/soa/esb/listeners/deployers/mc/test_esb_archives");
        final VirtualFile archiveFile = VFS.getVirtualFile(dir, archiveName);
        deployment = VFSDeploymentFactory.getInstance().createVFSDeployment(archiveFile);
        mainDeployer = (MainDeployerImpl) getBean("MainDeployer");
        mainDeployer.deploy(deployment);
    }
    
    public void testJBossWebMetaDataCreated() throws Throwable
    {
        final DeploymentUnit unit = mainDeployer.getDeploymentUnit(deployment.getName());
        final JBossWebMetaData webMD = unit.getAttachment(JBossWebMetaData.class);
        assertNotNull(webMD);
        
        final JBossServletsMetaData servlets = webMD.getServlets();
        assertEquals(1, servlets.size());
        
        final String servletClassName = servlets.iterator().next().getServletClass();
        assertEquals("esb.ws.FirstServiceESB.SimpleListener.Implementation", servletClassName);
        
        final List<ServletMappingMetaData> mappings = webMD.getServletMappings();
        assertEquals(1, mappings.size());
        assertEquals("FirstServiceESB_SimpleListener", mappings.get(0).getServletName());
        assertEquals(1, mappings.get(0).getUrlPatterns().size());
        assertEquals("/ebws/FirstServiceESB/SimpleListener", mappings.get(0).getUrlPatterns().get(0));
    }
    
    public void testServletAddedToClassPath() throws Throwable
    {
        final VFSDeploymentUnit unit = (VFSDeploymentUnit) mainDeployer.getDeploymentUnit(deployment.getName());
        final VirtualFile classesDir = getWebInfClassesDir(unit.getClassPath());
        
        assertNotNull(classesDir.getChild("esb/ws/FirstServiceESB/SimpleListener/Implementation.class"));
    }
    
    public void testUndeployCleanup() throws DeploymentException, MalformedURLException, IOException
    {
        final EsbWebServiceDeployer wsDeployer = (EsbWebServiceDeployer) getBean("EsbWebServiceDeployer");
        
        final VFSDeploymentUnit unit = (VFSDeploymentUnit) mainDeployer.getDeploymentUnit(deployment.getName());
        final URL inMemRootUrl = VfsUtil.createInMemUrl("exploded-esb-archive");
        final VirtualFile inMemRootDir = VfsUtil.createInMemoryFs(inMemRootUrl);
        unit.addAttachment(EsbWebServiceDeployer.DYNAMIC_CL_DIR_KEY, inMemRootDir);
        unit.addAttachment(EsbWebServiceDeployer.DYNAMIC_CL_URL_KEY, inMemRootUrl);
        
        wsDeployer.undeploy(unit, null);
        
        assertNull(unit.getAttachment(EsbWebServiceDeployer.DYNAMIC_CL_DIR_KEY));
        assertNull(unit.getAttachment(EsbWebServiceDeployer.DYNAMIC_CL_URL_KEY));
        
        final VirtualFile classesDir = getWebInfClassesDir(unit.getClassPath());
        assertNull(classesDir);
    }
    
    public void tearDown() throws Exception
    {
        mainDeployer.undeploy(deployment);
        super.tearDown();
    }

    private VirtualFile getWebInfClassesDir(final List<VirtualFile> classpath)
    {
        for (VirtualFile file : classpath)
        {
            if (file.getPathName().equals("WEB-INF/classes"))
            {
                return file;
            }
        }
        return null;
    }
    
}
