/*
* JBoss, Home of Professional Open Source
* Copyright 2006, JBoss Inc., and individual contributors as indicated
* by the @authors tag. See the copyright.txt in the distribution for a
* full listing of individual contributors.
*
* This is free software; you can redistribute it and/or modify it
* under the terms of the GNU Lesser General Public License as
* published by the Free Software Foundation; either version 2.1 of
* the License, or (at your option) any later version.
*
* This software is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public
* License along with this software; if not, write to the Free
* Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
* 02110-1301 USA, or see the FSF site: http://www.fsf.org.
*/
package org.jboss.soa.esb.lifecycle;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;

import java.util.ArrayList;
import java.util.List;

import junit.framework.JUnit4TestAdapter;

import org.junit.Test;

/**
 * @author Kevin Conner
 *
 */
public class LifecycleResourceManagerUnitTest {
    
    @Test
    public void testAssociation() 
        throws Exception
    {
        // create a lifecycle resource
        final LifecycleResource<TestLifecycleResource1> resource = new LifecycleResource<TestLifecycleResource1>(new TestLifecycleResource1Factory()) ;
        // get unassociated resource and check id
        final TestLifecycleResource1 first = resource.getLifecycleResource() ;
        assertEquals(LifecycleResourceManager.DEFAULT_IDENTITY, first.getId()) ;
        assertNull(first.getDestroyId()) ;
        
        
        // associate deployment
        final String deploymentName = "deploymentName" ;
        // get associated resource and check id
        LifecycleResourceManager.getSingleton().associateDeployment(deploymentName) ;
        final TestLifecycleResource1 second = resource.getLifecycleResource() ;
        final String id = second.getId() ;
        assertFalse(LifecycleResourceManager.DEFAULT_IDENTITY.equals(id)) ;
        assertNull(second.getDestroyId()) ;
        assertNull(first.getDestroyId()) ;
        
        // disassociate deployment
        LifecycleResourceManager.getSingleton().disassociateDeployment(deploymentName) ;
        // check id has been cleaned
        assertEquals(id, second.getDestroyId()) ;
        assertNull(first.getDestroyId()) ;
        // check resources are eventually cleaned up.
        LifecycleResourceManager.getSingleton().cleanupAllResources() ;
        assertEquals(LifecycleResourceManager.DEFAULT_IDENTITY, first.getDestroyId()) ;
    }
    
    // Check order
    // create three resources (none, 0, 1000)
    // disassociate deployment
    // check id has been cleaned and resources destroyed in correct order.
    
    @Test
    public void testOrder()
        throws Exception
    {
        // create factories and lifecycle resources
        final TestLifecycleResource2Factory factory1 = new TestLifecycleResource2Factory() ;
        final LifecycleResource<TestLifecycleResource2> lifecycleResource1 = new LifecycleResource<TestLifecycleResource2>(factory1) ;
        final TestLifecycleResource2Factory factory2 = new TestLifecycleResource2Factory() ;
        final LifecycleResource<TestLifecycleResource2> lifecycleResource2 = new LifecycleResource<TestLifecycleResource2>(factory2, 100) ;
        final TestLifecycleResource2Factory factory3 = new TestLifecycleResource2Factory() ;
        final LifecycleResource<TestLifecycleResource2> lifecycleResource3 = new LifecycleResource<TestLifecycleResource2>(factory3, 0) ;
        
        // create resources
        final TestLifecycleResource2 resource1 = lifecycleResource1.getLifecycleResource() ;
        final TestLifecycleResource2 resource2 = lifecycleResource2.getLifecycleResource() ;
        final TestLifecycleResource2 resource3 = lifecycleResource3.getLifecycleResource() ;
        
        // check correct factory is used
        assertEquals(factory1, resource1.getFactory()) ;
        assertEquals(factory2, resource2.getFactory()) ;
        assertEquals(factory3, resource3.getFactory()) ;
        
        // check nothing is destroyed
        assertEquals(0, TestLifecycleResource2Factory.getDestroyOrder().size()) ;
        
        // cleanup all resources
        LifecycleResourceManager.getSingleton().cleanupAllResources() ;
        
        // check we have three resources destroyed
        assertEquals(3, TestLifecycleResource2Factory.getDestroyOrder().size()) ;
        
        // check order of destroy calls.
        assertEquals(resource1, TestLifecycleResource2Factory.getDestroyOrder().get(0)) ;
        assertEquals(resource3, TestLifecycleResource2Factory.getDestroyOrder().get(1)) ;
        assertEquals(resource2, TestLifecycleResource2Factory.getDestroyOrder().get(2)) ;
    }
    
    private static class TestLifecycleResource1
    {
        private final String id ;
        private String destroyId ;
        
        TestLifecycleResource1(final String id)
        {
            this.id = id ;
        }
        
        String getId()
        {
            return id ;
        }
        
        String getDestroyId()
        {
            return destroyId ;
        }
        
        void setDestroyId(final String destroyId)
        {
            this.destroyId = destroyId ;
        }
    }
    
    private static class TestLifecycleResource1Factory implements LifecycleResourceFactory<TestLifecycleResource1>
    {
        public TestLifecycleResource1 createLifecycleResource(final String lifecycleIdentity)
            throws LifecycleResourceException
        {
            return new TestLifecycleResource1(lifecycleIdentity) ;
        }

        public void destroyLifecycleResource(final TestLifecycleResource1 resource,
            final String lifecycleIdentity)
            throws LifecycleResourceException
        {
            resource.setDestroyId(lifecycleIdentity) ;
        }
    }
    
    private static class TestLifecycleResource2
    {
        private final LifecycleResourceFactory<TestLifecycleResource2> factory ;
        
        TestLifecycleResource2(final LifecycleResourceFactory<TestLifecycleResource2> factory)
        {
            this.factory = factory ;
        }
        
        LifecycleResourceFactory<TestLifecycleResource2> getFactory()
        {
            return factory ;
        }
    }
    
    private static class TestLifecycleResource2Factory implements LifecycleResourceFactory<TestLifecycleResource2>
    {
        private static final List<TestLifecycleResource2> destroyOrder = new ArrayList<TestLifecycleResource2>() ;
        
        public TestLifecycleResource2 createLifecycleResource(final String lifecycleIdentity)
            throws LifecycleResourceException
        {
            return new TestLifecycleResource2(this) ;
        }

        public void destroyLifecycleResource(final TestLifecycleResource2 resource,
            final String lifecycleIdentity)
            throws LifecycleResourceException
        {
            destroyOrder.add(resource) ;
        }
        
        static List<TestLifecycleResource2> getDestroyOrder()
        {
            return destroyOrder ;
        }
    }
    
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(LifecycleResourceManagerUnitTest.class);
    }
}
