/*
 * JBoss, Home of Professional Open Source
 * Copyright 2007, JBoss Inc., and others contributors as indicated
 * by the @authors tag. All rights reserved.
 * See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 * This copyrighted material is made available to anyone wishing to use,
 * modify, copy, or redistribute it subject to the terms and conditions
 * of the GNU Lesser General Public License, v. 2.1.
 * This program is distributed in the hope that it will be useful, but WITHOUT A
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
 * PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more details.
 * You should have received a copy of the GNU Lesser General Public License,
 * v.2.1 along with this distribution; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA  02110-1301, USA.
 *
 * (C) 2005-2007,
 */

package org.jboss.soa.esb.actions.scripting;

import junit.framework.TestCase;

import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionLifecycleException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.message.Message;
import org.jboss.soa.esb.message.format.MessageFactory;

/**
 * Unit tests for ScriptingAction.
 * <p/>
 * Based on {@link GroovyActionProcessorUnitTest} by Tom Fennelly.
 *
 * @author dward at jboss.org
 */
public class ScriptingActionUnitTest extends TestCase
{

	public void test_bsh_file()
		throws ConfigurationException, ActionLifecycleException, ActionProcessingException
	{
		do_script_file("test.bsh");
	}

	private void do_script_file(String script)
		throws ConfigurationException, ActionLifecycleException, ActionProcessingException
	{
		ConfigTree config = new ConfigTree("<config/>");
		config.setAttribute("script", "/org/jboss/soa/esb/actions/scripting/" + script);
		String messageContents = "Hello World!";
		config.setAttribute("messageContents", messageContents);
		ScriptingAction action = new ScriptingAction(config);
		Message message = MessageFactory.getInstance().getMessage();
		assertProcessingOK(action, message, messageContents);
	}

	public void test_bsh_inlined()
		throws ConfigurationException, ActionLifecycleException, ActionProcessingException
	{
		do_script_inlined(
				"message.getBody().add(config.getAttribute(\"messageContents\").getBytes());",
				"bsh" );
	}

	public void test_beanshell_inlined()
		throws ConfigurationException, ActionLifecycleException, ActionProcessingException
	{
		do_script_inlined(
				"message.getBody().add(config.getAttribute(\"messageContents\").getBytes());",
				"beanshell" );
	}

	private void do_script_inlined(String script, String language)
		throws ConfigurationException, ActionLifecycleException, ActionProcessingException
	{
		ConfigTree config = new ConfigTree("<config/>");
		config.setAttribute("language", language);
		config.setAttribute("supportMessageBasedScripting", "true");
		String messageContents = "Hello World - Inlined!";
		config.setAttribute("messageContents", messageContents);
		ScriptingAction action = new ScriptingAction(config);
		Message message = MessageFactory.getInstance().getMessage();
		message.getBody().add(script);
		assertProcessingOK(action, message, messageContents);
	}

	private void assertProcessingOK(ScriptingAction action, Message message, String messageContents)
		throws ActionLifecycleException, ActionProcessingException
	{
		action.initialise();
		action.process(message);
		byte[] bodyContents = (byte[]) message.getBody().get();
		assertNotNull("Expected body contents to be set.", bodyContents);
		assertEquals( messageContents, new String(bodyContents) );
	}

	public void test_unregistered_language()
		throws ConfigurationException
	{
		ConfigTree config = new ConfigTree("<config/>");
		config.setAttribute("language", "unregistered");
		config.setAttribute("script", "/org/jboss/soa/esb/actions/scripting/test.bsh"); // doesn't matter
		ScriptingAction action = new ScriptingAction(config);
		try
		{
			action.initialise();
			fail("ScriptingAction with unregistered language should throw an ActionLifecycleException.");
		}
		catch (ActionLifecycleException ale)
		{
			assertEquals( "language 'unregistered' not registered", ale.getMessage() );
		}
	}

}
