/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.soa.esb.actions.routing;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;
import static org.junit.Assert.fail;

import java.io.Serializable;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.net.URI;
import java.net.URISyntaxException;

import javax.jms.Destination;
import javax.jms.ExceptionListener;
import javax.jms.JMSException;
import javax.jms.Message;
import javax.jms.MessageProducer;
import javax.jms.QueueConnection;
import javax.jms.QueueSession;
import javax.jms.Session;
import javax.jms.TextMessage;
import javax.naming.Context;
import javax.naming.InitialContext;
import javax.naming.NamingException;

import junit.framework.JUnit4TestAdapter;

import org.apache.log4j.Logger;
import org.jboss.internal.soa.esb.rosetta.pooling.ConnectionException;
import org.jboss.soa.esb.ConfigurationException;
import org.jboss.soa.esb.actions.ActionProcessingException;
import org.jboss.soa.esb.addressing.eprs.JMSEpr;
import org.jboss.soa.esb.common.Environment;
import org.jboss.soa.esb.helpers.ConfigTree;
import org.jboss.soa.esb.helpers.NamingContextException;
import org.jboss.soa.esb.helpers.NamingContextPool;
import org.jboss.soa.esb.listeners.gateway.DefaultESBPropertiesSetter;
import org.jboss.soa.esb.message.format.MessageFactory;
import org.jboss.soa.esb.notification.jms.JMSPropertiesSetter;
import org.jboss.soa.esb.testutils.SerializableMockQueue;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.mockejb.jms.MockQueue;
import org.mockejb.jms.ObjectMessageImpl;
import org.mockejb.jms.QueueConnectionFactoryImpl;
import org.mockejb.jms.TextMessageImpl;
import org.mockejb.jndi.MockContextFactory;

/**
 * JMSRouter unit tests.
 * 
 * @author <a href="mailto:kevin.conner@jboss.com">Kevin Conner</a>
 */
public class JMSRouterUnitTest
{
    private static final String CONNECTION_FACTORY = "ConnectionFactory" ;
    private static final String QUEUE_NAME = "failQueue" ;
    private static final String PROPERTY_SETTER_TEST_NAME = "MockJMSPropertiesSetter" ;
    private final String messageID = "1234-junittest";
    private final String bodyContent = "hello";
    private org.jboss.soa.esb.message.Message msg;
    private ConfigTree tree;
    
    @Before
    public void setUp()
        throws Exception
    {
        MockContextFactory.setAsInitial();
        
        final Context ctx = NamingContextPool.getNamingContext(null);
        try
        {
            ctx.rebind(CONNECTION_FACTORY, new MockQueueConnectionFactory());
            ctx.rebind(QUEUE_NAME, new MockQueue(QUEUE_NAME));
        }
        finally
        {
            NamingContextPool.releaseNamingContext(ctx) ;
        }
        System.setProperty(Environment.JNDI_SERVER_CONTEXT_FACTORY, System.getProperty(Context.INITIAL_CONTEXT_FACTORY)) ;
    }
    
    @After
    public void tearDown()
        throws Exception
    {
        MockContextFactory.revertSetAsInitial();
    }
    
    @Before
    public void setup() throws URISyntaxException
    {
        msg = createESBMessageObject( messageID, bodyContent );
        tree = createConfigTree();
    }
    
    @Test
    public void testRetry()
        throws Exception
    {
        final ConfigTree config = new ConfigTree("config") ;
        config.setAttribute("jndiName", QUEUE_NAME);
        
        final JMSRouter router = new JMSRouter(config) ;
        
        assertEquals("Original connection count", 1, MockQueueConnectionFactory.queueConnectionCount) ;
        assertEquals("Original session count", 1, MockQueueExceptionHandlerInvocationHandler.queueSessionCount) ;
        assertEquals("Original producer count", 1, MockQueueSessionInvocationHandler.producerCount) ;
        
        try
        {
            router.process(MessageFactory.getInstance().getMessage()) ;
            fail("Expected to receive an ActionProcessingException") ;
        }
        catch (final ActionProcessingException ape) {} // expected
        
        // Connection count changes as we fire the exception handler and this closes connections/sessions/producers
        assertEquals("Original connection count", 2, MockQueueConnectionFactory.queueConnectionCount) ;
        // session count should increment by one to reflect the second session created when handling the exception.
        assertEquals("Original session count", 2, MockQueueExceptionHandlerInvocationHandler.queueSessionCount) ;
        // One producer for setup, first failure and second retry.
        assertEquals("Original producer count", 3, MockQueueSessionInvocationHandler.producerCount) ;
    }
    
    @Test
    public void processWithUnwrapFalse() throws ConfigurationException, NamingException, JMSException, ActionProcessingException, URISyntaxException
    {
        MockJMSRouter router = new MockJMSRouter(tree);
        router.route( msg );

        assertProcessContract( messageID, msg, router );
    }
    
    @Test
    public void processWithUnwrapTrue() throws ConfigurationException, NamingException, JMSException, ActionProcessingException, URISyntaxException
    {
        tree.setAttribute( "unwrap", "true" );
        MockJMSRouter router = new MockJMSRouter( tree );

        router.route( msg );

        assertProcessContract( messageID, msg, router );
        final javax.jms.Message jmsMessage = ((MockJMSRouter) router).getJmsMessage();

        assertTrue ( jmsMessage instanceof TextMessage );
        final javax.jms.TextMessage textMessage = (TextMessage) jmsMessage;

        assertEquals ( textMessage.getJMSCorrelationID(), messageID );
        assertEquals ( textMessage.getText(), bodyContent );
    }
    
    @Test
    public void constructorWithSecurity() throws ConfigurationException, NamingException, JMSException, ActionProcessingException, URISyntaxException
    {
        final String principal = "guest";
        final String credential = "guest";
        tree.setAttribute( "unwrap", "true" );
        tree.setAttribute( JMSRouter.SECURITY_PRINCIPAL, principal );
        tree.setAttribute( JMSRouter.SECURITY_CREDITIAL, credential );
        MockJMSRouter router = new MockJMSRouter( tree );
        
        assertEquals (  principal, router.getSecurityPrincipal() );
        assertEquals (  credential, router.getSecurityCredential() );
    }
    
    @Test ( expected = ConfigurationException.class )
    public void shouldThrowIfPrincipalIsNull() throws ConfigurationException, NamingException, JMSException, ActionProcessingException, URISyntaxException
    {
        tree.setAttribute( JMSRouter.SECURITY_CREDITIAL, "testpassword" );
        new MockJMSRouter( tree );
    }
    
    @Test ( expected = ConfigurationException.class )
    public void shouldThrowIfCredentialIsNull() throws ConfigurationException, NamingException, JMSException, ActionProcessingException, URISyntaxException
    {
        tree.setAttribute( JMSRouter.SECURITY_PRINCIPAL, "testuser" );
        new MockJMSRouter( tree );
    }

    @Test ( expected = ActionProcessingException.class )
    public void shouldThrowIfObjectIsNotAMessageObject() throws ConfigurationException, NamingException, JMSException, ActionProcessingException
    {
        ConfigTree tree = createConfigTree();
        JMSRouter router = new MockJMSRouter(tree);
        router.route( "test" );
    }

    @Test
    public void setJMSReplyToQueue() throws JMSException, URISyntaxException, ConfigurationException, NamingException, ConnectionException, NamingContextException
    {
        try
        {
            MockContextFactory.setAsInitial();
            Context context = new InitialContext();
            MockContextFactory.setDelegateContext( context );
        
            final MockQueue queue = new SerializableMockQueue( QUEUE_NAME );
        
            context.rebind( QUEUE_NAME, queue );

            // inbound is to simulate a jms message arriving at a gateway endpoint.
            TextMessageImpl inBoundJmsMessage = new TextMessageImpl();
            inBoundJmsMessage.setJMSReplyTo( queue );
            new DefaultESBPropertiesSetter().setPropertiesFromJMSMessage( inBoundJmsMessage, msg );
        
            JMSRouter router = new JMSRouter( createConfigTree() );
            
            // outbound is to simulate a new jms message that is about to leave the ESB.
            TextMessageImpl outBoundJmsMessage = new TextMessageImpl();
            router.setJMSReplyTo( outBoundJmsMessage,  msg );
        }
        finally
        {
            MockContextFactory.revertSetAsInitial();
        }
    }
    
    @Test
    public void constructWithDefaultPersitentAttribute() throws ConfigurationException, NamingException, JMSException
    {
        ConfigTree config = createConfigTree();
        JMSRouter router = new JMSRouter( config );
        assertTrue( router.isDeliveryModePersistent() );
    }

    @Test
    public void constructWithPersitentAttribute() throws ConfigurationException, NamingException, JMSException
    {
        ConfigTree config = createConfigTree();
        config.setAttribute( JMSRouter.PERSISTENT_ATTR, "false" );
        JMSRouter router = new JMSRouter( config );

        assertFalse ( router.isDeliveryModePersistent() );
    }

    @Test
    public void constructWithDefaultPriorityAttribute() throws ConfigurationException, NamingException, JMSException
    {
        ConfigTree config = createConfigTree();
        JMSRouter router = new JMSRouter( config );
        assertEquals( javax.jms.Message.DEFAULT_PRIORITY, router.getPriority() );
    }

    @Test
    public void constructWithPriorityAttribute() throws ConfigurationException, NamingException, JMSException
    {
        final int expectedPriority = 9;
        ConfigTree config = createConfigTree();
        config.setAttribute( JMSRouter.PRIORITY_ATTR, String.valueOf( expectedPriority ) );
        JMSRouter router = new JMSRouter( config );

        assertEquals ( expectedPriority, router.getPriority() );
    }

    @Test
    public void constructWithDefaultTimeToLiveAttribute() throws ConfigurationException, NamingException, JMSException
    {
        ConfigTree config = createConfigTree();
        JMSRouter router = new JMSRouter( config );
        assertEquals( javax.jms.Message.DEFAULT_TIME_TO_LIVE, router.getTimeToLive() );
    }

    @Test
    public void constructWithTimeToLiveAttribute() throws ConfigurationException, NamingException, JMSException
    {
        final long ttl = 6000l;
        ConfigTree config = createConfigTree();
        config.setAttribute( JMSRouter.TIME_TO_LIVE_ATTR, String.valueOf( ttl ) );
        JMSRouter router = new JMSRouter( config );

        assertEquals ( ttl, router.getTimeToLive() );
    }

    @Test
    public void specifyJNDIPropertiesInConfig() throws JMSException, URISyntaxException, ConfigurationException, NamingException, ConnectionException
    {
        final String jndiContextFactoryName = "org.jnp.interfaces.NamingContextFactory";
        final String jndiURL = "localhost:1099";
        final String jndiPkgPrefix = "org.jboss.naming";
        final String connectionFactory = "ConnectionFactory";

        ConfigTree config = createConfigTree();
        config.setAttribute( JMSEpr.JNDI_CONTEXT_FACTORY_TAG, jndiContextFactoryName );
        config.setAttribute( JMSEpr.JNDI_URL_TAG, jndiURL );
        config.setAttribute( JMSEpr.JNDI_PKG_PREFIX_TAG, jndiPkgPrefix );
        config.setAttribute( JMSEpr.CONNECTION_FACTORY_TAG, connectionFactory );

        MockJMSRouter router = new MockJMSRouter( config );
        assertEquals( jndiContextFactoryName, router.getContextFactoryName() );
        assertEquals( jndiURL, router.getJndiURL() );
        assertEquals( jndiPkgPrefix, router.getJndiPkgPrefix() );
        assertEquals( connectionFactory, router.getConnectionFactory() );
    }

    @Test
    public void usePropertyStrategy() throws ConfigurationException, NamingException, JMSException, ActionProcessingException
    {
        final ConfigTree tree = createConfigTree() ;
        tree.setAttribute(JMSRouter.PROPERTY_STRATEGY, MockJMSPropertiesSetter.class.getName()) ;
        final MockJMSRouter router = new MockJMSRouter(tree) ;
        router.route(msg) ;
        assertEquals("Property setter value", PROPERTY_SETTER_TEST_NAME, router.getJmsMessage().getStringProperty(PROPERTY_SETTER_TEST_NAME)) ;
    }
    
    private void assertProcessContract( final String messageID, final org.jboss.soa.esb.message.Message msg, JMSRouter router ) throws ActionProcessingException, JMSException
    {
        final org.jboss.soa.esb.message.Message message = router.process ( msg );
        assertNull ( "Routers process should return null",  message );
        final javax.jms.Message jmsMessage = ((MockJMSRouter) router).getJmsMessage();
        assertEquals ( jmsMessage.getJMSCorrelationID(), messageID );
    }
    
    private org.jboss.soa.esb.message.Message createESBMessageObject( final String messageID, final String body) throws URISyntaxException
    {
        msg = MessageFactory.getInstance().getMessage();
        msg.getHeader().getCall().setMessageID( new URI ( "1234-junittest" ) );
        msg.getBody().add(body.getBytes());
        return msg;
    }

    private static ConfigTree createConfigTree()
    {
        ConfigTree tree = new ConfigTree("test");
        tree.setAttribute("jndiName", QUEUE_NAME);
        return tree;
    }

    private static final class MockQueueConnectionFactory extends QueueConnectionFactoryImpl
    {
        static int queueConnectionCount ;
        
        @Override
        public QueueConnection createQueueConnection() throws JMSException
        {
            queueConnectionCount++ ;
            return (QueueConnection)Proxy.newProxyInstance(getClass().getClassLoader(), new Class[] {QueueConnection.class},
                    new MockQueueExceptionHandlerInvocationHandler(super.createQueueConnection())) ;
        }
    }
    
    private static final class MockQueueExceptionHandlerInvocationHandler implements InvocationHandler
    {
        private final QueueConnection queueConnection ;
        private ExceptionListener exceptionListener ;
        static int queueSessionCount ;
            
        MockQueueExceptionHandlerInvocationHandler(final QueueConnection queueConnection)
        {
            this.queueConnection = queueConnection ;
        }
            
        public Object invoke(final Object proxy, final Method method, final Object[] args)
            throws Throwable
        {
            final String methodName = method.getName() ;
            if ("setExceptionListener".equals(methodName))
            {
                exceptionListener = (ExceptionListener)args[0] ;
                return null ;
            }
            else if ("getExceptionListener".equals(methodName))
            {
                return exceptionListener ;
            }
            else
            {
                final Object response = method.invoke(queueConnection, args) ;
                if (response instanceof QueueSession)
                {
                    queueSessionCount++ ;
                    final QueueSession queueSession = (QueueSession)response ;
                    return (QueueSession)Proxy.newProxyInstance(getClass().getClassLoader(), new Class[] {QueueSession.class},
                            new MockQueueSessionInvocationHandler(this, queueSession)) ;
                }
                else
                {
                    return response ;
                }
            }
        }
        
        void fireExceptionListener(final JMSException exception)
        {
            if (exceptionListener != null)
            {
                exceptionListener.onException(exception) ;
            }
        }
    }
    
    private static final class MockQueueSessionInvocationHandler implements InvocationHandler
    {
        private final MockQueueExceptionHandlerInvocationHandler exceptionHandler ;
        private final QueueSession queueSession ;
        static int producerCount ;
            
        MockQueueSessionInvocationHandler(final MockQueueExceptionHandlerInvocationHandler exceptionHandler, final QueueSession queueSession)
        {
            this.exceptionHandler = exceptionHandler ;
            this.queueSession = queueSession ;
        }
            
        public Object invoke(final Object proxy, final Method method, final Object[] args)
            throws Throwable
        {
            final String methodName = method.getName() ;
            if ("recover".equals(methodName))
            {
                return null ;
            }
            else if ("createProducer".equals(methodName))
            {
                producerCount++ ;
                return new MockFailMessageProducer(exceptionHandler) ;
            }
            else if ("getAcknowledgeMode".equals(methodName))
            {
                return Integer.valueOf(Session.AUTO_ACKNOWLEDGE) ;
            }
            else if ("getTransacted".equals(methodName))
            {
                return Boolean.FALSE ;
            }
            else
            {
                return method.invoke(queueSession, args) ;
            }
        }
    }
    
    private static final class MockFailMessageProducer implements MessageProducer
    {
        private final MockQueueExceptionHandlerInvocationHandler exceptionHandler ;
        
        public MockFailMessageProducer(final MockQueueExceptionHandlerInvocationHandler exceptionHandler)
        {
            this.exceptionHandler = exceptionHandler ;
        }
        
        public void close() throws JMSException {}

        public int getDeliveryMode()
            throws JMSException
        {
            return 0;
        }

        public Destination getDestination()
            throws JMSException
        {
            return null;
        }

        public boolean getDisableMessageID()
            throws JMSException
        {
            return false;
        }

        public boolean getDisableMessageTimestamp()
            throws JMSException
        {
            return false;
        }

        public int getPriority()
            throws JMSException
        {
            return 0;
        }

        public long getTimeToLive()
            throws JMSException
        {
            return 0;
        }

        public void send(Message arg0)
            throws JMSException
        {
            exception() ;
        }

        public void send(Destination arg0, Message arg1)
            throws JMSException
        {
            throw new JMSException("Deliberate send exception") ;
        }

        public void send(Message arg0, int arg1, int arg2, long arg3)
            throws JMSException
        {
            throw new JMSException("Deliberate send exception") ;
        }

        private void exception()
            throws JMSException
        {
            final JMSException exception = new JMSException("Deliberate send exception") ;
            exceptionHandler.fireExceptionListener(exception) ;
            throw exception ;
        }
        
        public void send(Destination arg0, Message arg1, int arg2, int arg3, long arg4)
            throws JMSException
        {
        }

        public void setDeliveryMode(int arg0) throws JMSException {}

        public void setDisableMessageID(boolean arg0) throws JMSException {}

        public void setDisableMessageTimestamp(boolean arg0) throws JMSException {}

        public void setPriority(int arg0) throws JMSException {}

        public void setTimeToLive(long arg0) throws JMSException {}
    }
    
    private static class MockJMSRouter extends JMSRouter
    {
        @SuppressWarnings ( "unused" )
        private Logger log = Logger.getLogger( MockJMSRouter.class );

        private javax.jms.Message jmsMessage;
        
        private String securityPrincipal;
        private String securityCredential;

        public MockJMSRouter(ConfigTree propertiesTree) throws ConfigurationException, NamingException, JMSException
        {
            super( propertiesTree );
        }

        @Override
        protected void createQueueSetup( String queueName ) throws ConfigurationException { }

        @Override
        protected void createQueueSetup( String queueName,
            String jndiContextFactory,
            String jndiUrl,
            String jndiPkgPrefix,
            String connectionFactory,
            String securityPrincipal,
            String securityCredential) throws ConfigurationException
        {
            this.securityPrincipal = securityPrincipal;
            this.securityCredential = securityCredential;
        }
        
        @Override
        protected void send( javax.jms.Message jmsMessage ) throws JMSException
        {
            this.jmsMessage = jmsMessage;
        }

        public javax.jms.Message getJmsMessage()
        {
            return jmsMessage;
        }

        @Override
        protected javax.jms.Message createJMSMessageWithObjectType( Object objectFromBody ) throws JMSException
        {
            TextMessageImpl textMessage = new TextMessageImpl();
            textMessage.setText( new String ((byte[])objectFromBody) );
            return textMessage;
        }

        @Override
        protected javax.jms.Message createObjectMessage( Object message ) throws JMSException
        {
            ObjectMessageImpl impl = new ObjectMessageImpl();
            impl.setObject( (Serializable) message );
            return impl;
        }
        
        public String getSecurityPrincipal() 
        {
            return securityPrincipal;
        }

        public String getSecurityCredential() 
        {
            return securityCredential;
        }
    }
    
    static class MockJMSPropertiesSetter implements JMSPropertiesSetter
    {
        public void setJMSProperties(final org.jboss.soa.esb.message.Message esbMsg, final javax.jms.Message jmsMessage) throws JMSException
        {
            jmsMessage.setStringProperty(PROPERTY_SETTER_TEST_NAME, PROPERTY_SETTER_TEST_NAME) ;
        }
    }
    
    public static junit.framework.Test suite()
    {
        return new JUnit4TestAdapter(JMSRouterUnitTest.class);
    }
}
