/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jboss.internal.soa.esb.listeners.war;

import junit.framework.TestCase;
import org.jboss.internal.soa.esb.util.StreamUtils;
import org.jboss.internal.soa.esb.util.XMLHelper;
import org.jboss.soa.esb.util.ClassUtil;

/**
 * @author <a href="mailto:tom.fennelly@jboss.com">tom.fennelly@jboss.com</a>
 */
public class WebDeploymentArchiveUnitTest extends TestCase {

    public void test_empty() throws Exception {
        WebDeploymentArchive modelArch = new WebDeploymentArchive("testarch.esb");

        String webXML = modelArch.getWebXML();
        String expected = StreamUtils.readStreamString(ClassUtil.getResourceAsStream("empty.xml", getClass()), "UTF-8") ;
        assertTrue("XML comparison", XMLHelper.compareXMLContent(expected, webXML)) ;
    }

    public void test_plain_servlet() throws Exception {
        WebDeploymentArchive modelArch = new WebDeploymentArchive("testarch.esb");
        Servlet servlet = new Servlet("servletX", "com.acme.XServlet", modelArch.getWebModel());

        servlet.getUrlMappings().add("/x/*");

        String webXML = modelArch.getWebXML();
        String expected = StreamUtils.readStreamString(ClassUtil.getResourceAsStream("plain_servlet.xml", getClass()), "UTF-8") ;
        assertTrue("XML comparison", XMLHelper.compareXMLContent(expected, webXML)) ;
    }

    public void test_plain_servlet_and_filter() throws Exception{
        WebDeploymentArchive modelArch = new WebDeploymentArchive("testarch.esb");
        Servlet servlet = new Servlet("servletX", "com.acme.XServlet", modelArch.getWebModel());

        servlet.getUrlMappings().add("/x/*");

        // Just need to create the filter instance...
        new Filter("servletXFilter", "com.acme.XServletFilter", servlet);

        String webXML = modelArch.getWebXML();
        String expected = StreamUtils.readStreamString(ClassUtil.getResourceAsStream("plain_servlet_and_filter.xml", getClass()), "UTF-8") ;
        assertTrue("XML comparison", XMLHelper.compareXMLContent(expected, webXML)) ;
    }

    public void test_with_method_constraints_no_roles() throws Exception {
        WebDeploymentArchive modelArch = new WebDeploymentArchive("testarch.esb");
        Servlet servlet = new Servlet("servletX", "com.acme.XServlet", modelArch.getWebModel());
        SecurityConstraints securityConstraints = new SecurityConstraints(servlet);

        servlet.getUrlMappings().add("/x/*");

        securityConstraints.getProtectedMethods().add("PUT");
        securityConstraints.getProtectedMethods().add("POST");
        securityConstraints.getProtectedMethods().add("DELETE");

        String webXML = modelArch.getWebXML();
        String expected = StreamUtils.readStreamString(ClassUtil.getResourceAsStream("with_method_constraints_no_roles.xml", getClass()), "UTF-8") ;
        assertTrue("XML comparison", XMLHelper.compareXMLContent(expected, webXML)) ;
    }

    public void test_with_role_constraints_no_methods() throws Exception {
        WebDeploymentArchive modelArch = new WebDeploymentArchive("testarch.esb");
        Servlet servlet = new Servlet("servletX", "com.acme.XServlet", modelArch.getWebModel());
        SecurityConstraints securityConstraints = new SecurityConstraints(servlet);

        servlet.getUrlMappings().add("/x/*");

        securityConstraints.getAllowedRoles().add("manager");
        securityConstraints.getAllowedRoles().add("accountant");

        String webXML = modelArch.getWebXML();
        String expected = StreamUtils.readStreamString(ClassUtil.getResourceAsStream("with_role_constraints_no_methods.xml", getClass()), "UTF-8") ;
        assertTrue("XML comparison", XMLHelper.compareXMLContent(expected, webXML)) ;
    }

    public void test_with_role_constraints_no_methods_with_guarantee() throws Exception {
        WebDeploymentArchive modelArch = new WebDeploymentArchive("testarch.esb");
        Servlet servlet = new Servlet("servletX", "com.acme.XServlet", modelArch.getWebModel());
        SecurityConstraints securityConstraints = new SecurityConstraints(servlet);

        servlet.getUrlMappings().add("/x/*");

        securityConstraints.getAllowedRoles().add("manager");
        securityConstraints.getAllowedRoles().add("accountant");

        securityConstraints.setTransportGuarantee("INTEGRAL");

        String webXML = modelArch.getWebXML();
        String expected = StreamUtils.readStreamString(ClassUtil.getResourceAsStream("with_role_constraints_no_methods_with_guarantee.xml", getClass()), "UTF-8") ;
        assertTrue("XML comparison", XMLHelper.compareXMLContent(expected, webXML)) ;
    }

    public void test_with_authmethod() throws Exception {
        WebDeploymentArchive modelArch = new WebDeploymentArchive("testarch.esb");

        modelArch.getWebModel().setAuthMethod("BASIC");

        String webXML = modelArch.getWebXML();
        String webExpected = StreamUtils.readStreamString(ClassUtil.getResourceAsStream("with_authmethod_webxml.xml", getClass()), "UTF-8") ;
        assertTrue("XML comparison", XMLHelper.compareXMLContent(webExpected, webXML)) ;

        String jbossWebXML = modelArch.getJBossWebXML();
        String jbossWebExpected = StreamUtils.readStreamString(ClassUtil.getResourceAsStream("with_authmethod_jbosswebxml.xml", getClass()), "UTF-8") ;
        assertTrue("XML comparison", XMLHelper.compareXMLContent(jbossWebExpected, jbossWebXML)) ;
    }

    public void test_with_authdomain() throws Exception {
        WebDeploymentArchive modelArch = new WebDeploymentArchive("testarch.esb");

        modelArch.getWebModel().setAuthDomain("java:/jaas/JBossWS");

        String webXML = modelArch.getWebXML();
        String webExpected = StreamUtils.readStreamString(ClassUtil.getResourceAsStream("with_authdomain_webxml.xml", getClass()), "UTF-8") ;
        assertTrue("XML comparison", XMLHelper.compareXMLContent(webExpected, webXML)) ;

        String jbossWebXML = modelArch.getJBossWebXML();
        String jbossWebExpected = StreamUtils.readStreamString(ClassUtil.getResourceAsStream("with_authdomain_jbosswebxml.xml", getClass()), "UTF-8") ;
        assertTrue("XML comparison", XMLHelper.compareXMLContent(jbossWebExpected, jbossWebXML)) ;
    }
}
